/***************************************************************************
 *   Copyright (C) 2006 by Weintraub, Benkard, Van Roy and Jeziorski       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/***********************************************************
 * Computes non-stationary equilibrium for given oe        *
 *                                                         *
 * INPUT:                                                  *
 *    1) Output filename - string                          *
 *    2) Control parameters - vector                       *
 *    3) Profit function parameters - vector               *
 *    4) Profit function parameters for Ipopt - vector     *
 *    5) State transition parameters - vector              *
 *    6) Model Constants - vector                          *
 *    7) Model Initial values - vector                     *
 *    8) Convergence tolerance values - vector             *
 *  OUTPUT:                                                *
 *    1) xmax - scalar                                     *
 *    2) Value function - vector                           *
 *    3) Iota - vector                                     *
 *    4) Rho - vector                                      *
 *    5) Entry rate - scalar                               *
 *    6) Tildes - vector                                   *
 *    7) Probabily of continuing - vector                  *
 *    8) Equilibrium prices - vector                       *
 *    9) Market shares - vector                            *
 *   10) Profit - vector                                   *
 *   11) Initial condition used to calculate equilibrium   *
 *          profits in the last iteration                  *
 *   12) Concetration ratios - value                       *
 *   13) 1 - if success, 0 - if failed                     *
 ***********************************************************/

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <math.h>
#include <compoe/compoe.h>
#include <compoe/compoe_nonstat.h>
#include <mat.h>
#include <string.h>
#define NAME_LENGTH 30

mxArray *readvar(MATFile *mfp, const char *var_name);

ofstream myfile;

int main(int argc, char *argv[]) {
  if(argc<3) {
    cout << "The executable needs two arguments: output and input .mat file names" << endl;
    exit(1);
  }
  char filename[NAME_LENGTH];
  MATFile *mfp;

  strncpy(filename, argv[1], NAME_LENGTH);

  if ((mfp = matOpen(filename, "r")) == NULL) {
    #ifdef COUT_DEBUG
      cout << "Cannot open MAT-file " << filename << endl;
    #endif
    exit(1);
  } else {
    #ifdef COUT_DEBUG
      cout << "Opening MAT-file " << filename << endl;
    #endif
  }

  mxArray *var = readvar(mfp, "filename");

  char *file=new char[mxGetN(var)+1];
  mxGetString(var, file, mxGetN(var)+1);

  myfile.open(file);

  mxArray *controlmx = readvar(mfp, "control");
  mxArray *prof_funmx = readvar(mfp, "prof_fun");
  mxArray *prof_fun_ipoptmx = readvar(mfp, "prof_fun_ipopt");
  mxArray *transitionmx = readvar(mfp, "transition");
  mxArray *constantsmx = readvar(mfp, "constants");
  mxArray *initmx = readvar(mfp, "init");
  mxArray *convtolmx = readvar(mfp, "convtol");
  mxArray *boundsmx = readvar(mfp, "bounds");

  if (matClose(mfp) != 0) {
    printf("Error closing file %s\n",filename);
    return(EXIT_FAILURE);
  }

  double *control = mxGetPr(controlmx);
  double *prof_fun = mxGetPr(prof_funmx);
  double *prof_fun_ipopt = mxGetPr(prof_fun_ipoptmx);
  double *transition = mxGetPr(transitionmx);
  double *constants = mxGetPr(constantsmx);
  double *init = mxGetPr(initmx);
  double *convtol = mxGetPr(convtolmx);
  double *bounds = mxGetPr(boundsmx);
  int xmax=int(init[0]);

  #ifdef DEBUG
  myfile << "Oblivious Equilibrium Solver" << endl;
  myfile << "----------------------------" << endl << endl;

  myfile << "\nProfit Function Parameters:" << endl;
  for(int ii=0;ii<mxGetM(prof_funmx);ii++) myfile << "  " << ii << ": " << prof_fun[ii] << endl;

  myfile << "\nProfit Function Parameters for Ipopt:" << endl;
  for(int ii=0;ii<mxGetM(prof_fun_ipoptmx);ii++) myfile << "  " << ii << ": " << prof_fun_ipopt[ii] << endl;

  myfile << "\n\nDynamic Parameters:\n\n";
  for(int ii=0;ii<mxGetM(constantsmx);ii++) myfile << "  " << ii << ": " << constants[ii] << endl;

  myfile << "\n\nTransition Parameters:\n\n";
  for(int ii=0;ii<mxGetM(transitionmx);ii++) myfile << "  " << ii << ": " << transition[ii] << endl;

  if(control[2]>0) {
    myfile << "  Fixed Number of Firms (No entry/No exit): " << control[2] << endl;
  } else {
    if(control[1]==1) {
        myfile << "  Poisson Entry Process" << endl;
    } else {
        myfile << "  Constant Entry Process" << endl;
    }
  }

  myfile << "\nParameters:\n\n";
  myfile << "  xmax: " << xmax << endl;
  #endif
  CompOE *oe = new CompOE();
  oe->setOutput(&myfile);

  VectorMTL<double> iotaInit(xmax);
  VectorMTL<double> rhoInit(xmax);

  iotaInit=init[1];
  rhoInit=init[2];
  int success;
  if(mxGetM(controlmx)>3) {
    oe->setMethod(int(control[3]));
  }

  strncpy(filename, argv[2], NAME_LENGTH);
  if ((mfp = matOpen(filename, "w")) == NULL) {
    #ifdef COUT_DEBUG
      cout << "Cannot open MAT-file " << filename << endl;
    #endif
    #ifdef DEBUG
      myfile << "Cannot open MAT-file " << filename << endl;
    #endif
    exit(1);
  } else {
    #ifdef COUT_DEBUG
      cout << "Opening MAT-file " << filename << endl;
    #endif
    #ifdef DEBUG
      myfile << "Opening MAT-file " << filename << endl;
    #endif
  }

  oe->initialize(control, prof_fun, prof_fun_ipopt, transition, constants, init, convtol);
  success=oe->compute(xmax,rhoInit,iotaInit);
  mxArray *output = mxCreateDoubleMatrix(1,1,mxREAL);
  double *outputpr = mxGetPr(output);
  outputpr[0] = success;
  matPutVariable(mfp, "success", output);

  if(success==0) {
    myfile << "Oblivious Equilibrium calculation failed\n";
    delete oe;

    myfile.close();
    delete[] file;

    if (matClose(mfp) != 0) {
      printf("Error closing file %s\n",file);
      return(EXIT_FAILURE);
    }
    return 0;
  } else {
    // Compute profit at the expected state

    VectorMTL<double> *tildes = oe->getTildes();
    VectorMTL<double> *prices_end = oe->getPrices();

    VectorMTL<double> prices(xmax),ms(xmax);
    VectorMTL<double> profstate(xmax);
    double prodsur,conssur;
    
    // Compute profit at the expected state
    oe->compProfAndStats_extra(*tildes,*prices_end,prices,profstate,ms,prodsur,conssur);

    double totalms=sum(ms*(*tildes));
    ms/=totalms;

    double hhi50=0;
    for(int x=0;x<xmax;x++) {
      hhi50+=10000*ms[x]*ms[x]*(*tildes)[x];
    }
    VectorMTL<double> *temp_v;

    // Last initial condition for profit
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = prices_end;
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "pricesInit", output);

    // Profit
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = &profstate;
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "profit", output);

    // Market shares
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = &ms;
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "ms", output);

    // Prices
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = &prices;
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "prices", output);

    // Probability of continuing in the industry
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = oe->getPrcont();;
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "prcont", output);

    // Value function
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = oe->getV();
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "V", output);

    // Entry rate
    output = mxCreateDoubleMatrix(1,1,mxREAL);
    outputpr = mxGetPr(output);
    outputpr[0]=oe->getLambda();
    matPutVariable(mfp, "lambda", output); 
    
    // Tildes
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = oe->getTildes();
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "tildes", output);

    // Rho
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = oe->getRho();
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "rho", output);

    // Iota
    output = mxCreateDoubleMatrix(1,xmax,mxREAL);
    outputpr = mxGetPr(output);
    temp_v = oe->getIota();
    memcpy(outputpr,temp_v->getData(),xmax*sizeof(double));
    matPutVariable(mfp, "iota", output);

    // xmax
    output = mxCreateDoubleMatrix(1,1,mxREAL);
    outputpr = mxGetPr(output);
    outputpr[0]=xmax;
    matPutVariable(mfp, "xmax", output);

    // HHI
    output = mxCreateDoubleMatrix(1,1,mxREAL);
    outputpr = mxGetPr(output);
    outputpr[0]=hhi50;
    matPutVariable(mfp, "hhi50", output);

    // Producer surplus
    output = mxCreateDoubleMatrix(1,1,mxREAL);
    outputpr = mxGetPr(output);
    outputpr[0]=prodsur;
    matPutVariable(mfp, "prodsur", output);

    // Consumer surplus
    output = mxCreateDoubleMatrix(1,1,mxREAL);
    outputpr = mxGetPr(output);
    outputpr[0]=conssur;
    matPutVariable(mfp, "conssur", output);
  }
  delete oe;

  myfile.close();
  delete[] file;

  if (matClose(mfp) != 0) {
    printf("Error closing file %s\n",filename);
    return(EXIT_FAILURE);
  }

  return 0;
}

mxArray *readvar(MATFile *mfp, const char *var_name) {
    mxArray *array_ptr;
    int *from, *to;

    if ((array_ptr = matGetVariable(mfp, var_name)) == NULL) {
        cout << "Cannot read matlab variable " << var_name << endl;
        myfile << "Cannot read matlab variable " << var_name << endl;
        exit(1);
    }
    if ((mxGetNumberOfElements(array_ptr) == 1) && mxIsNumeric(array_ptr)) {
      #ifdef COUT_DEBUG
        cout << "Reading matlab variable " << var_name << "=" << mxGetScalar(array_ptr) << endl;
      #endif
      #ifdef DEBUG
        myfile << "Reading matlab variable " << var_name << "=" << mxGetScalar(array_ptr) << endl;
      #endif
    } else {
        #ifdef COUT_DEBUG
          cout << "Reading matlab variable " << var_name << " (";
        #endif
        #ifdef DEBUG
          myfile << "Reading matlab variable " << var_name << " (";
        #endif
        from = (int *) mxGetDimensions(array_ptr);
        to = from+mxGetNumberOfDimensions(array_ptr)-1;
        for (; from<to; from++) {
        #ifdef COUT_DEBUG
          cout << *from << "x";
        #endif
        #ifdef DEBUG
          myfile << *from << "x";
        #endif
        }
        #ifdef COUT_DEBUG
          cout << *from << ")"<< endl;
        #endif
        #ifdef DEBUG
          myfile << *from << ")"<< endl;
        #endif
    }

    return array_ptr;
}
