/***********************************************************
 * Computes non-stationary equilibrium for given oe        *
 *                                                         *
 * INPUT:                                                  *
 *    1) Output filename - string                          *
 *    2) Control parameters - vector                       *
 *    3) Profit function parameters - vector               *
 *    4) Profit function parameters for ipopt - vector     *
 *    5) State transition parameters - vector              *
 *    6) Model Constants - vector                          *
 *    7) Model Initial values - vector                     *
 *    8) Convergence tolerance values - vector             *
 *    9) Non-stationary eq. specific parameters - vector   *
 *  Oblivious equlibrium (vectors must be of size xmax):   *
 *   10) Lambda - int                                      *
 *   11) Iota - vector                                     *
 *   12) Rho - vector                                      *
 *   13) Value function - vector                           *
 *   14) Initial Prices - vector                           *
 *   15) Initial state - vector                            *
 *   16) OE tildes - vector
 *                                                         *
 * OUTPUT:                                                 *
 *    1) Value function - Matrix (Tbar+1,xmax)             *
 *    2) Iota - Matrix                                     *
 *    3) Rho - Matrix                                      *
 *    4) Lambda - vector (Tbar)                            *
 *    5) s - Matrix (Tbar+1,xmax)                          *
 *    6) Profit matrix - Matrix (Tbar,xmax)                *
 *    7) pMatrix - Matrix (Tbar,xmax)                      *
 ***********************************************************/

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <math.h>
#include <compoe/compoe.h>
#include <compoe/compoe_nonstat.h>
#include <mat.h>
#include <string.h>
#define NAME_LENGTH 100

mxArray *readvar(MATFile *mfp, const char *var_name);

ofstream myfile;


int main(int argc, char *argv[]) {
  if(argc<3) {
    cout << "The executable needs two arguments: output and input .mat file names" << endl;
    exit(1);
  }
  char filename[NAME_LENGTH];
  MATFile *mfp;

  strncpy(filename, argv[1], NAME_LENGTH);

  if ((mfp = matOpen(filename, "r")) == NULL) {
    #ifdef COUT_DEBUG
      cout << "Cannot open MAT-file " << filename << endl;
    #endif
    exit(1);
  } else {
    #ifdef COUT_DEBUG
      cout << "Opening MAT-file " << filename << endl;
    #endif
  }

  mxArray *var = readvar(mfp, "filename");

  char *file=new char[mxGetN(var)+1];
  mxGetString(var, file, mxGetN(var)+1);

  myfile.open(file);

  mxArray *controlmx = readvar(mfp, "control");
  mxArray *prof_funmx = readvar(mfp, "prof_fun");
  mxArray *prof_fun_ipoptmx = readvar(mfp, "prof_fun_ipopt");
  mxArray *transitionmx = readvar(mfp, "transition");
  mxArray *constantsmx = readvar(mfp, "constants");
  mxArray *initmx = readvar(mfp, "init");
  mxArray *convtolmx = readvar(mfp, "convtol");
  mxArray *boundsmx = readvar(mfp, "bounds");
  mxArray *nonstatparamsmx = readvar(mfp, "nonstatparams");

  double lambda_oe = mxGetPr(readvar(mfp, "lambda"))[0];

  mxArray *iotamx = readvar(mfp, "iota");
  mxArray *rhomx = readvar(mfp, "rho");
  mxArray *vmx = readvar(mfp, "V");
  mxArray *pricesmx = readvar(mfp, "prices");
  mxArray *s0mx = readvar(mfp, "s0");
  mxArray *tildesmx = readvar(mfp, "tildes");

  MatrixMTL<double> *A;
  int shockn;
  if (matGetVariable(mfp, "A") == NULL) {
    A = new MatrixMTL<double>(1,1);
    (*A)[0][0]=1;
    shockn=1;
  } else {
    mxArray *Amx = readvar(mfp, "A");
    double *Apr = mxGetPr(Amx);
    shockn = mxGetN(Amx);

    A = new MatrixMTL<double>(shockn,shockn);
    int z=0;
    for(int i=0;i<shockn;++i) {
      for(int j=0;j<shockn;++j) {
        (*A)[j][i]=Apr[z];
        z++;
      }
    }
  }

  VectorMTL<double> D0(shockn);
  if(shockn==1) {
    D0[0]=1;
  } else {
    mxArray *D0mx = readvar(mfp, "D0");
    double *D0pr = mxGetPr(D0mx);
    for(int i=0;i<shockn;++i) {
      D0[i]=D0pr[i];
    }
  }

  if (matClose(mfp) != 0) {
    printf("Error closing file %s\n",filename);
    return(EXIT_FAILURE);
  }

  double *control = mxGetPr(controlmx);
  double *prof_fun = mxGetPr(prof_funmx);
  double *prof_fun_ipopt = mxGetPr(prof_fun_ipoptmx);
  double *transition = mxGetPr(transitionmx);
  double *constants = mxGetPr(constantsmx);
  double *init = mxGetPr(initmx);
  double *convtol = mxGetPr(convtolmx);
  double *bounds = mxGetPr(boundsmx);
  double *nonstatparams = mxGetPr(nonstatparamsmx);
  int xmax = int(nonstatparams[12]);
  int Tbar = int(nonstatparams[10]);

  int size = mxGetN(iotamx);
  if(size!=xmax) {
    myfile << "Size of iota is " << size << ". Should be " << xmax << ". Exiting..." << endl;
    printf("Wrong size of iota\n");
    exit(1);
  }
  size = mxGetN(rhomx);
  if(size!=xmax) {
    myfile << "Size of rho is " << size << ". Should be " << xmax << ". Exiting..." << endl;
    printf("Wrong size of rho\n");
    exit(1);
  }
  size = mxGetN(vmx);
  if(size!=xmax) {
    myfile << "Size of V is " << size << ". Should be " << xmax << ". Exiting..." << endl;
    printf("Wrong size of V\n");
    exit(1);
  }
  size = mxGetN(pricesmx);
  if(size!=xmax) {
    myfile << "Size of prices is " << size << ". Should be " << xmax << ". Exiting..." << endl;
    printf("Wrong size of prices\n");
    exit(1);
  }

  size = mxGetN(s0mx);
  if(size!=xmax) {
    myfile << "Size of s0 is " << size << ". Should be " << xmax << ". Exiting..." << endl;
    printf("Wrong size of s0\n");
    exit(1);
  }

  VectorMTL<double> iota_oe(xmax);
  VectorMTL<double> rho_oe(xmax);
  VectorMTL<double> V_oe(xmax);
  VectorMTL<double> s0_init(xmax);
  VectorMTL<double> prices_oe(xmax);
  VectorMTL<double> tildes(xmax);

  for(int i=0;i<xmax;i++) {
    iota_oe[i]=(mxGetPr(iotamx))[i];
    rho_oe[i]=(mxGetPr(rhomx))[i];
    V_oe[i]=(mxGetPr(vmx))[i];
    prices_oe[i]=(mxGetPr(pricesmx))[i];
    s0_init[i]=(mxGetPr(s0mx))[i];
    tildes[i]=(mxGetPr(tildesmx))[i];
  }

  myfile << "Non-stationary OE Solver" << endl;
  myfile << "----------------------------" << endl << endl;

  myfile << "\nProfit Function Parameters:" << endl;
  for(int ii=0;ii<mxGetM(prof_funmx);ii++) myfile << "  " << ii << ": " << prof_fun[ii] << endl;

  myfile << "\nProfit Function Parameters for Ipopt:" << endl;
  for(int ii=0;ii<mxGetM(prof_fun_ipoptmx);ii++) myfile << "  " << ii << ": " << prof_fun_ipopt[ii] << endl;
 
  myfile << "\n\nDynamic Parameters:\n\n";
  for(int ii=0;ii<mxGetM(constantsmx);ii++) myfile << "  " << ii << ": " << constants[ii] << endl;

  myfile << "\n\nTransition Parameters:\n\n";
  for(int ii=0;ii<mxGetM(transitionmx);ii++) myfile << "  " << ii << ": " << transition[ii] << endl;

  if(control[2]>0) {
    myfile << "  Fixed Number of Firms (No entry/No exit): " << control[2] << endl;
  } else {
    if(control[1]==1) {
        myfile << "  Poisson Entry Process" << endl;
    } else {
        myfile << "  Constant Entry Process" << endl;
    }
  }

  myfile << "\nParameters:\n\n";
  myfile << "  xmax: " << xmax << endl;
  myfile << "  Tbar: " << Tbar << endl;
  myfile << "  Entry Rate: " << lambda_oe << endl;

  CompOE_nonstat *oe_nonstat = new CompOE_nonstat;
  oe_nonstat->setOutput(&myfile);

  oe_nonstat->initialize(control, prof_fun, prof_fun_ipopt, transition, constants, init, 
      convtol, nonstatparams, A);

  /*** Initialize vector size for subtracting ***/
  int xmax_subtract;
  if(nonstatparams[18]!=2) {
    xmax_subtract=1;
  } else {
    xmax_subtract=xmax;
  }

  MatrixMTL<double> rhoMatrix0((Tbar+1)*xmax_subtract,xmax);
  MatrixMTL<double> iotaMatrix0((Tbar+1)*xmax_subtract,xmax);
  VectorMTL<double> lambda0((Tbar+1)*xmax_subtract);
  MatrixMTL<double> pMatrix((Tbar+1)*xmax_subtract,xmax);

  /*** Innitialize calulations ***/
  for(int x=0;x<xmax_subtract;++x) {
    for(int j=0;j<=Tbar;j++) {
      int i=x*(Tbar+1)+j;
      rhoMatrix0.setRow(i,rho_oe);
      iotaMatrix0.setRow(i,iota_oe);
      pMatrix.setRow(i,prices_oe);
    }
  }

  lambda0 = lambda_oe;

  oe_nonstat->computeStatsNonstat(bounds,Tbar,xmax,iotaMatrix0,rhoMatrix0,lambda0,pMatrix,s0_init,V_oe,tildes,D0);

  strncpy(filename, argv[2], NAME_LENGTH);
  if ((mfp = matOpen(filename, "w")) == NULL) {
    #ifdef COUT_DEBUG
      cout << "Cannot open MAT-file " << filename << endl;
    #endif
    #ifdef DEBUG
      myfile << "Cannot open MAT-file " << filename << endl;
    #endif
    exit(1);
  } else {
    #ifdef COUT_DEBUG
      cout << "Opening MAT-file " << filename << endl;
    #endif
    #ifdef DEBUG
      myfile << "Opening MAT-file " << filename << endl;
    #endif
  }

  mxArray *output;
  double *outputpr;

  const char *field_names[] = {"cs", "ps", "ts", "c1", "c2", "c4", "c10", "c20", "hhi", "entrants", "investment", "firms_number"};
  mwSize dims[1] = {1};
  mxArray *statsOE = mxCreateStructArray(1, dims, 12, field_names);

  const char *subfield_names[] = {"avr", "pre", "var"};
  mxArray *statsOEsub;
  mxArray *field_value;
  double *field_value_pr;

  MatrixMTL<double> output_init(11,3);
  oe_nonstat->getStats(output_init);
  cout << output_init << endl;

  

  /* Fill in the one-dimentional fields */
  for(int i=0;i<11;i++) {
    /* Create a subarray */
    statsOEsub = mxCreateStructArray(1, dims, 3, subfield_names);
    /* Plant a subarray */
    mxSetFieldByNumber(statsOE,0,i,statsOEsub);
    /* Fill in subarray */
    for(int j=0;j<3;j++) {
       /* Create value */
       field_value=mxCreateDoubleMatrix(1,1,mxREAL);
       field_value_pr=mxGetPr(field_value);
       /* Fill in value */
       field_value_pr[0]=output_init[i][j];
       /* Plant value */
       mxSetFieldByNumber(statsOEsub,0,j,field_value);
    }
  }

  cout << 1 <<endl;

  /* Expected number of firms */
  VectorMTL<double> firmsnumberavg(xmax);
  VectorMTL<double> firmsnumberpre(xmax);
  VectorMTL<double> firmsnumbervar(xmax);
 
  oe_nonstat->getFirmNumber(firmsnumberavg,firmsnumberpre,firmsnumbervar);

  cout << firmsnumberavg;
  statsOEsub = mxCreateStructArray(1, dims, 3, subfield_names);
  mxSetFieldByNumber(statsOE,0,11,statsOEsub);

  field_value = mxCreateDoubleMatrix(xmax,1,mxREAL);
  field_value_pr=mxGetPr(field_value);
  for(int i=0;i<xmax;i++) {
    field_value_pr[i]=firmsnumberavg[i];
  }
  mxSetFieldByNumber(statsOEsub,0,0,field_value);

  field_value = mxCreateDoubleMatrix(xmax,1,mxREAL);
  field_value_pr=mxGetPr(field_value);
  for(int i=0;i<xmax;i++) {
    field_value_pr[i]=firmsnumberpre[i];
  }
  mxSetFieldByNumber(statsOEsub,0,1,field_value);

  field_value = mxCreateDoubleMatrix(xmax,1,mxREAL);
  field_value_pr=mxGetPr(field_value);
  for(int i=0;i<xmax;i++) {
    field_value_pr[i]=firmsnumbervar[i];
  }
  mxSetFieldByNumber(statsOEsub,0,2,field_value);
  
  matPutVariable(mfp, "statsNOEresolv", statsOE);
  if(bounds[12]==1) {
    int M=bounds[11];

    statsOE = mxCreateStructArray(1, dims, 12, field_names);
    
    MatrixMTL<double> output_init_path(11*M,3);

    oe_nonstat->getStats_path(output_init_path, M);

    cout << output_init_path;
 
    int z=0;
    for(int i=0;i<11;i++) {
      statsOEsub = mxCreateStructArray(1, dims, 3, subfield_names);
      mxSetFieldByNumber(statsOE,0,i,statsOEsub);
      
      /* avg */
      mxArray *field_value1=mxCreateDoubleMatrix(1,M,mxREAL);
      double *field_value_pr1=mxGetPr(field_value1);

      /* var */
      mxArray *field_value2=mxCreateDoubleMatrix(1,M,mxREAL);
      double *field_value_pr2=mxGetPr(field_value2);
      
      /* pre */
      mxArray *field_value3=mxCreateDoubleMatrix(1,M,mxREAL);
      double *field_value_pr3=mxGetPr(field_value3);

      for(int k=0;k<M;++k,++z) {
        field_value_pr1[k]=output_init_path[z][0];
        field_value_pr2[k]=output_init_path[z][1];
        field_value_pr3[k]=output_init_path[z][2];
      }

      mxSetFieldByNumber(statsOEsub,0,0,field_value1);
      mxSetFieldByNumber(statsOEsub,0,1,field_value2);
      mxSetFieldByNumber(statsOEsub,0,2,field_value3);
    }
    

    matPutVariable(mfp, "statsNOEresolv_path", statsOE);
  }

  cout << "done\n";

  delete A;
  delete oe_nonstat;

  myfile.close();
  delete[] file;

  if (matClose(mfp) != 0) {
    printf("Error closing file %s\n",filename);
    return(EXIT_FAILURE);
  }

  return 0;

}

mxArray *readvar(MATFile *mfp, const char *var_name) {
    mxArray *array_ptr;
    int *from, *to;

    if ((array_ptr = matGetVariable(mfp, var_name)) == NULL) {
        cout << "Cannot read matlab variable " << var_name << endl;
        myfile << "Cannot read matlab variable " << var_name << endl;
        exit(1);
    }
    if ((mxGetNumberOfElements(array_ptr) == 1) && mxIsNumeric(array_ptr)) {
      #ifdef COUT_DEBUG
        cout << "Reading matlab variable " << var_name << "=" << mxGetScalar(array_ptr) << endl;
      #endif
      #ifdef DEBUG
        myfile << "Reading matlab variable " << var_name << "=" << mxGetScalar(array_ptr) << endl;
      #endif
    } else {
        #ifdef COUT_DEBUG
          cout << "Reading matlab variable " << var_name << " (";
        #endif
        #ifdef DEBUG
          myfile << "Reading matlab variable " << var_name << " (";
        #endif
        from = (int *) mxGetDimensions(array_ptr);
        to = from+mxGetNumberOfDimensions(array_ptr)-1;
        for (; from<to; from++) {
        #ifdef COUT_DEBUG
          cout << *from << "x";
        #endif
        #ifdef DEBUG
          myfile << *from << "x";
        #endif
        }
        #ifdef COUT_DEBUG
          cout << *from << ")"<< endl;
        #endif
        #ifdef DEBUG
          myfile << *from << ")"<< endl;
        #endif
    }

    return array_ptr;
}
