/***************************************************************************
 *   Copyright (C) 2006 by Jeziorski, Weintraub, Benkard and Van Roy       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef MATRIX_H
#define MATRIX_H

/* Comment out this to switch off LAPACK */
//#define LAPACK
#include <sys/time.h>
#include<math.h>
#include<stdlib.h>
#include<iostream>
#include<vectorMTL.h>
#include<tridiagmatrix.h>
#include<sparsematrixMTL.h>

//#define LAPACK

using namespace std;

namespace MTL {

#ifdef LAPACK
  extern "C" {
    void dgetrf_(int*,int *,double *,int*,int*,int*);
    void dgetri_(int *n, double *a, int *lda, int *ipiv, 
      double *work, int *lwork, int *info);
    void dgetrs_(char *trans,int *n, int *nrhs, double *a, int *lda, int *ipiv, 
      double *b, int *ldb, int *info);
  }
#endif
//template <class T> class MatrixMTL;
//template <class T> class VectorMTL;
//template <class T> class TriDiagMatrixMTL;
//template <class T> class SparseMatrixMTL;

template <typename T> inline const MatrixMTL<T> operator+(const MatrixMTL<T>&, const MatrixMTL<T>&);
template <typename T> inline const MatrixMTL<T> operator+(const MatrixMTL<T>&, const T);
template <typename T> inline const MatrixMTL<T> operator+(const T, const MatrixMTL<T>&);
template <typename T> inline const MatrixMTL<T> operator-(const MatrixMTL<T>&, const MatrixMTL<T>&);
template <typename T> inline const MatrixMTL<T> operator-(const MatrixMTL<T>&, const T);
template <typename T> inline const MatrixMTL<T> operator-(const T, const MatrixMTL<T>&);
template <typename T> inline const MatrixMTL<T> operator*(const MatrixMTL<T>&, const MatrixMTL<T>&);
template <typename T> inline const MatrixMTL<T> operator*(const MatrixMTL<T>&, const T);
template <typename T> inline const MatrixMTL<T> operator*(const T, const MatrixMTL<T>&);
template <typename T> inline const MatrixMTL<T> operator/(const MatrixMTL<T>&, const MatrixMTL<T>&);
template <typename T> inline const MatrixMTL<T> operator/(const MatrixMTL<T>&, const T);
template <typename T> inline const MatrixMTL<T> operator/(const T, const MatrixMTL<T>&);
template <typename T> std::ostream& operator<<(std::ostream&, const MatrixMTL<T>&);

template <typename T> T norm(const MatrixMTL<T> &input);
template <typename T> T diffNorm(const MatrixMTL<T> &input1, const MatrixMTL<T> &input2);

template <typename T> MatrixMTL<T> mSqrt(const MatrixMTL<T> &);
template <typename T> MatrixMTL<T> mAbs(const MatrixMTL<T> &);

/**
	@author Przemyslaw Jeziorski <przemekj@stanford.edu>
*/


template <class T> class MatrixMTL {
  friend class TriDiagMatrixMTL<T>;
  friend class SparseMatrixMTL<T>;

  friend std::ostream& operator<< <T>(std::ostream &output, const MatrixMTL<T> &p);

  friend const MatrixMTL<T> operator+<T>(const MatrixMTL<T>&, const MatrixMTL<T>&);
  friend const MatrixMTL<T> operator+<T>(const MatrixMTL<T>&, const T);
  friend const MatrixMTL<T> operator+<T>(const T, const MatrixMTL<T>&);
  friend const MatrixMTL<T> operator-<T>(const MatrixMTL<T>&, const MatrixMTL<T>&);
  friend const MatrixMTL<T> operator-<T>(const MatrixMTL<T>&, const T);
  friend const MatrixMTL<T> operator-<T>(const T, const MatrixMTL<T>&);
  friend const MatrixMTL<T> operator*<T>(const MatrixMTL<T>&, const MatrixMTL<T>&);
  friend const MatrixMTL<T> operator*<T>(const MatrixMTL<T>&, const T);
  friend const MatrixMTL<T> operator*<T>(const T, const MatrixMTL<T>&);
  friend const MatrixMTL<T> operator/<T>(const MatrixMTL<T>&, const MatrixMTL<T>&);
  friend const MatrixMTL<T> operator/<T>(const MatrixMTL<T>&, const T);
  friend const MatrixMTL<T> operator/<T>(const T, const MatrixMTL<T>&);

  friend MatrixMTL<T> mSqrt<T>(const MatrixMTL<T> &);
  friend MatrixMTL<T> mAbs<T>(const MatrixMTL<T> &);

private:
  int length, height,size;

public:
  T *data;
  inline MatrixMTL(int heightInput, int lengthInput) {
    height=heightInput;
    length=lengthInput;
    size=length*height;
    data = new T[size];

// No init of data
//    for (int i=0;i<size;i++) {
//      data[i]=0;
//    }
  }

  inline MatrixMTL(int heightInput, int lengthInput, T *data_init) {
    height=heightInput;
    length=lengthInput;
    size=length*height;
    data = data_init;
  }


  // copy constructor
  inline MatrixMTL(const MatrixMTL<T> &input) {
    height=input.height;
    length=input.length;
    size=input.size;
    data = new T[size];

    for(int i=0;i<size;i++) {
      data[i]=input.data[i];
    }
  }

  MatrixMTL(const SparseMatrixMTL<T> &p) {
    height=p.nrows;
    length=p.ncolumns;
    size=length*height;
    data = new T[size];
    int k=0,m=0;
    for (int i=0; i<p.nrows; i++) {
      for (int j=0; j<p.ncolumns; j++) {
        if ((k<p.size) && (p.columns[k]==j) && (p.rows[k]==i)) {
          data[m]=p.data[k];
          k++;
          m++;
        } else {
          data[m]=0;
          m++;
        }
      }
    }
  }

  MatrixMTL(const TriDiagMatrixMTL<T> &input);

  inline T *operator[](const int input) {
    return &(data[input*length]);
  }

  MatrixMTL<T> operator-() {
    MatrixMTL<T> output(height,length);

    for (int i=0; i<size; i++) {
      output.data[i]=-data[i];
    }

    return output;
  }

  inline MatrixMTL<T> &operator *=(const MatrixMTL<T> &input) {
    for (int i=0; i<size; i++) {
     data[i]*=input.data[i];
    }

    return *this;
  }

  inline MatrixMTL<T> &operator *=(T input) {
    for (int i=0; i<size; i++) {
      data[i]*=input;
    }

    return *this;
  }

  inline MatrixMTL<T> &operator /=(const MatrixMTL<T> &input) {
    for (int i=0; i<size; i++) {
     data[i]/=input.data[i];
    }

    return *this;
  }

  inline MatrixMTL<T> &operator /=(T input) {
    for (int i=0; i<size; i++) {
      data[i]/=input;
    }

    return *this;
  }

  inline MatrixMTL<T> &operator +=(const MatrixMTL<T> &input) {
    for (int i=0; i<size; i++) {
     data[i]+=input.data[i];
    }

    return *this;
  }

  inline MatrixMTL<T> &operator +=(const T input) {
    for (int i=0; i<size; i++) {
     data[i]+=input;
    }

    return *this;
  }

  inline MatrixMTL<T> &operator -=(const MatrixMTL<T> &input) {
    for (int i=0; i<size; i++) {
     data[i]-=input.data[i];
    }

    return *this;
  }

  inline MatrixMTL<T> &operator -=(T input) {
    for (int i=0; i<size; i++) {
      data[i]-=input;
    }

    return *this;
  }

  inline MatrixMTL<T> &operator=(const T input) {
    for (int i=0; i<size; i++) {
      data[i]=input;
    }
    return *this;
  }

  inline MatrixMTL<T> &operator=(const MatrixMTL<T> &input) {
    for (int i=0; i<size; i++) {
      data[i]=input.data[i];
    }
    return *this;
  }

  void eye();

  inline int getLength() {
    return length;
  }

  inline int getHeight() {
    return height;
  }

  inline int getSize() {
    return size;
  }

  /**
  	Norm of the matrix
        \param input
   */
  friend T norm<T>(const MatrixMTL<T> &);

  /**
  	Norm of the difference of two matrices
	\param input1
        \param input2
   */
  friend T diffNorm<T>(const MatrixMTL<T> &input1, const MatrixMTL<T> &input2);

  VectorMTL<T> getColumn(const int input);
  const VectorMTL<T> getRow(const int input);

  void setRow(const int input, const VectorMTL<T> col) {
    int offset = input*length;
    for(int i=0;i<length;i++) {
      data[offset+i]=col.data[i];
    }
  }

  void setColumn(const int input, const VectorMTL<T> col) {
    for(int i=0, j=input;i<height;i++,j+=length) {
      data[j]=col.data[i];
    }
  }

  inline T getData(const int input) {
    return data[input];
  }

  inline T* getData() { 
    return data;
  }
  inline void setData(const int input, const T dataInput) {
    data[input]=dataInput;
  }
  inline void setData(T *dataInput) {
    data=dataInput;
  }

  void extraRow(const int input) {
    T *data1 = data;
    data = new T[(height+1)*length];
    for(int i=0;i<height*length;i++) {
      data[i]=data1[i];
    }
    height++;
    delete[] data1;
  }

  void addDiag(const T element) {
    for(int i=0;i<size;i+=(length+1)) {
      data[i]+=element;
    }
  }
  /**
	Compute stationary distribution
        \param expfreq
   */

  int statDistr(VectorMTL<T> &expfreq);

  int inverse(MatrixMTL<T> &output) {
    if(!inverseNP(output)) {
      struct timeval t1;
      gettimeofday(&t1, 0);
      srand48(t1.tv_sec);
      int offset_i=0;
      for(int i=0;i<height;i++) {
        data[offset_i+i]+=drand48()/1e3;
        offset_i+=length;
      }
      inverseNP(output);
      cout << "MatrixMTL perturbed" << endl;
    }
    return 1;
  }

  int inverseNP(MatrixMTL<T> &output) {
  #ifndef LAPACK
    VectorMTL<T> temp(height);
    VectorMTL<T> idx(height);
    MatrixMTL<T> decomposition(height,length);

    if(!lu(decomposition,idx)) {
      cout << "Singularity problem in LU (matrix inverse routine)\n";
      return 0;
    } else {
      //      cout << "decomposition: " << decomposition;
      for(int i=0;i<length;i++) {
        temp.base(i);

        if(!decomposition.solveFromLU(idx,temp)) {
  //        decomposition.solveFromLU(idx,temp);
          cout << "Singularity problem when solving from LU" << endl;
          return 0;
        }
        int offset = 0; 
        for(int j=0; j<height; j++) {
          output.data[offset+i]=temp.data[j];
          offset+=length;
        }
      }
    }
    return 1;
  #else
    for(int i=0;i<size;i++) {
      output.data[i]=data[i];
    }
    int *b = new int[length];
    double *work = new double[3*size];
    int lwork = 3*size;
    int info;
    int lda=length;
    dgetrf_(&length,&height,output.data,&lda,b,&info); 
    dgetri_(&height,output.data,&lda,b,work,&lwork,&info);
    delete[] b;
    delete[] work;
    return 1;
  #endif
  }

  void transpose() {
    int offset1;
    int offset2;
    T temp;
    for(int i=0; i<height; i++) {
      offset1 = i*length;
      offset2 = height*(i+1)+i;
      for(int j=i+1; j<length; j++) {
        temp = data[offset1+j];
        data[offset1+j]=data[offset2];
        data[offset2]=temp;
//        cout << offset1+j << " " << offset2 <<endl;
//        getchar();
        offset2+=height;
      }
    }
  }

  MatrixMTL<T> dot(SparseMatrixMTL<T> &input) {
    MatrixMTL<T> output(height,input.ncolumns);
    output=0;
    for(int i=0;i<input.size;i++) {
      int offset1 = input.rows[i];
      int offset2 = input.columns[i];
      for(int j=0;j<height;j++) {
        output.data[offset2]+=data[offset1]*input.data[i];
        offset1+=length;
        offset2+=input.ncolumns;
      }
    }
    return output;
  }

  void switchRows(const int x,const int y);
  void switchColumns(const int x,const int y);

  int lu(MatrixMTL<T> &output, VectorMTL<T> &idx);
  int lu(MatrixMTL<T> &l, MatrixMTL<T> &u, VectorMTL<T> &idx);

  MatrixMTL<T> dot(MatrixMTL<T> &input);
  MatrixMTL<T> primedot(MatrixMTL<T> &input);
  MatrixMTL<T> dot(TriDiagMatrixMTL<T> &input);
  VectorMTL<T> dot(const VectorMTL<T> &input);
  VectorMTL<T> primedot(const VectorMTL<T> &input);

  int solve(VectorMTL<T> &b);
  int solveFromLU(VectorMTL<T> &idx, VectorMTL<T> &b);
  MatrixMTL<T> dotFactorial(const int input);

  ~MatrixMTL();
};

template <class T>
VectorMTL<T> MatrixMTL<T>::getColumn(const int input) {
  VectorMTL<T> output(height);

  for(int i=0;i<height;i++) {
    output.data[i]=data[i*length+input];
  }
  return output;
}

template <class T>
const VectorMTL<T> MatrixMTL<T>::getRow(const int input) {
  VectorMTL<T> output(length);

  int offset = length*input; 
  for(int i=0;i<length;i++) {
    output.data[i]=data[offset+i];
  }

  return output;
}


/***************/
/* unit matrix */
/***************/
template <class T>
void MatrixMTL<T>::eye() {
  (*this)=0;
  for(int i=0; i<height; i++) {
    (*this)[i][i]=1;
  }
}

template <class T>
MatrixMTL<T> MatrixMTL<T>::dot(MatrixMTL<T> &input) {
  int newHeight = height;
  int newLength = input.getLength();
  T sum;

  MatrixMTL<T> output(newHeight, newLength);

  for (int i=0; i<newHeight; i++) {
    for (int j=0; j<newLength; j++) {
      sum=0;
      for (int z=0; z<length; z++) {
        sum+=(*this)[i][z]*input[z][j];
      }
      output[i][j]=sum;
    }
  }
  return output;
}

template <class T>
MatrixMTL<T> MatrixMTL<T>::primedot(MatrixMTL<T> &input) {
  int newHeight = height;
  int newLength = input.getLength();
  T sum;

  MatrixMTL<T> output(newHeight, newLength);

  for (int i=0; i<newHeight; i++) {
    for (int j=0; j<newLength; j++) {
      sum=0;
      for (int z=0; z<length; z++) {
        sum+=(*this)[z][i]*input[z][j];
      }
      output[i][j]=sum;
    }
  }
  return output;
}

template <class T>
MatrixMTL<T> MatrixMTL<T>::dot(TriDiagMatrixMTL<T> &input) {
  MatrixMTL<T> output(height, length);

  // coumpute first column
  for (int i=0; i<height; i++) {
    output[i][0] = (*this)[i][0]*input.diag[0]+(*this)[i][1]*input.lower[0];
  }
  // coumpute middle columns 
  for (int i=0; i<height; i++) {
    for (int j=1; j<length-1; j++) {
      output[i][j] = (*this)[i][j-1]*input.upper[j-1]+(*this)[i][j]*input.diag[j]+(*this)[i][j+1]*input.lower[j];
    }
  }

  // coumpute last column
  for (int i=0; i<height; i++) {
    output[i][length-1] = (*this)[i][length-1]*input.diag[length-1]+(*this)[i][length-2]*input.upper[length-2];
  }

  return output;
}

template <class T>
VectorMTL<T> MatrixMTL<T>::dot(const VectorMTL<T> &input) {
  T sum=0;

  VectorMTL<T> output(height);
  int offset = 0;
  for (int i=0; i<height; i++) {
    sum=0;
    for(int j=0; j<length; j++) {
      sum+=data[offset+j]*input.data[j];
    }
    output.data[i]=sum;
    offset+=length;
  }
  return output;
}

template <class T>
VectorMTL<T> MatrixMTL<T>::primedot(const VectorMTL<T> &input) {
  T sum=0;

  VectorMTL<T> output(height);
  for (int i=0; i<length; i++) {
    sum=0;
    int offset=0;
    for(int j=0; j<height; j++) {
      sum+=data[i+offset]*input.data[j];
      offset+=length;
    }
    output.data[i]=sum;
  }
  return output;
}


template <class T>
void MatrixMTL<T>::switchRows(const int x, const int y) {
  int yRow=y*length;
  int xRow=x*length;
  T temp;
//  cout<<"switch rows (" << x<<","<<y<<")"<<endl;
  for (int i=0; i<length; i++) {
    temp=data[xRow+i];
    data[xRow+i]=data[yRow+i];
    data[yRow+i]=temp;
  }
}

// void MatrixMTL<T>::switchColumns(const int x, const int y) {
//   for (int i=0; i<height; i++) {
//     data[i]->switchElements(x,y);
//   }
// }
// 

template <class T>
int MatrixMTL<T>::lu(MatrixMTL<T>& output, VectorMTL<T> &idx) {
  T sum,temp;
  T pivot;
  int maxRow = 0;
/*  cout << "doing LU" << endl << endl;
  cout << *this;
  getchar();*/
  VectorMTL<T> vv(height);

  output=*this;
  for (int i=0;i<height;i++) {
    double big=0.0;
    for (int j=0;j<length;j++) {
        double temp;
        if ((temp=fabs(output[i][j])) > big) big=temp;
    }
    if (big == 0.0)  {
      cout << "Singular matrix in routine ludcmp" << endl;
//      big = 1e-5;
      return 0;
    }
    vv.data[i]=1.0/big;
  }


  for (int j=0; j<length; j++) { /* iteration on columns */
    int offset_i = 0;
    for (int i=0; i<j; i++) {
      sum=output.data[offset_i+j];

      int offset_k = 0;
      for(int k=0;k<i;k++) {
          sum-=output.data[offset_i+k]*output.data[offset_k+j];
          offset_k+=length;
      }
      output.data[offset_i+j]=sum;
/*      cout << "iteration beta " << i << "," << j << "   " << sum << endl;
      cout << output;
      getchar();*/
      offset_i+=length;
    }
    pivot = 0;
    offset_i=j*length;
    for (int i=j;i<height; i++) {
      sum=output.data[offset_i+j];

      int offset_k=0;
      for(int k=0;k<j;k++) {
          sum-=output.data[offset_i+k]*output.data[offset_k+j];
          offset_k+=length;
      }
      output.data[offset_i+j]=sum;

      if ( (temp=vv.data[i]*fabs(sum))>=pivot) {
        pivot = temp;
        maxRow = i;
      }
/*      cout << "iteration alpha " << i << "," << j << "   " << sum << endl;
      cout << output<< maxRow;
      getchar();*/
      offset_i+=length;
    }

    /* remember the permutation */
    idx.data[j]=maxRow;
//  cout << "doing LU2" << endl;

    /* pivoting */
    if (maxRow != j) {
      output.switchRows(maxRow,j);
      double change = vv.data[maxRow];
      vv.data[maxRow]=vv.data[j];
      vv.data[j]=change;
    }

    /* divite by the greatest element */
    offset_i=(j+1)*length;
    int offset_j=j*length;
    if (output.data[offset_j+j]==0) {
	cout << "singularity" << endl;
        output.data[offset_j+j]=1e-6;
//	return 0;
    }

    if(j!=length-1) {
      double den = 1/output.data[offset_j+j];
      for (int i=j+1;i<height; i++) {
        output.data[offset_i+j]*=den;
        offset_i+=length;
      }
    }
  }
  return 1;
}

template <class T>
int MatrixMTL<T>::lu(MatrixMTL<T> &l, MatrixMTL<T> &u, VectorMTL<T> &idx) {
  if (!lu(u,idx)) {
    return 0;
  } else {
    int offset_i=0;
    for (int i=0; i<length; i++) {
      l.data[offset_i+i]=1;
      int offset_j=0;
      for (int j=i+1; j<height; j++) {
        l.data[offset_j+i]=u.data[offset_j+i];
        u.data[offset_j+i]=0;
        offset_j+=length;
      }
    offset_i+=length; 
    }
  }
  return 1;
} 

template <class T>
int MatrixMTL<T>::solve(VectorMTL<T> &b) {
#ifndef LAPACK 
  VectorMTL<T> idx(height);
//  cout << height<<","<<length <<endl;
  MatrixMTL<T> decomposition(height,length);
//  cout << "w srodku solve" <<endl;
  int index;
  int heightLocal=height; // needed for the if statement
  int lengthLocal=length; // segfault otherwise
  T sum;

  if ( heightLocal!=lengthLocal ) {
    cout << "Not square matrix";
    exit(1);
  }

  if ((heightLocal=!b.getSize())) {
    cout << "Dimentions do not agree";
    exit(EXIT_FAILURE);
  }

  /* perform LU decomposition */
  if(!lu(decomposition,idx)) {
    cout << "LU decomposition failed in the linear solver routine" << endl;
    return 0;
  } else {
    //cout << "po LU" <<endl;

    int offset_i=0; 
    for (int i=0;i<height;i++) {
      if ((index=int(idx.data[i]))!=i) {
        b.switchElements(i,index);
      }
      sum=b.data[i];
      for(int j=0;j<=i-1;j++)
        sum-=decomposition.data[offset_i+j]*b.data[j];
      b.data[i]=sum;
      offset_i+=length;
    }
  }
  int offset_i = (height-1)*length;
  for (int i=height-1;i>=0;i--) {
    sum=b.data[i];
    for (int j=i+1;j<height;j++) {
      sum -= decomposition.data[offset_i+j]*b.data[j];
    }
    double temp;
    if((temp=decomposition.data[offset_i+i])==0) {
      cout << "Backward substitution in linear solver failed" << endl;
      temp = 1e-5;
    }
    b.data[i]=sum/temp;
    offset_i-=length;
  }
  return 1;
#else
//    cout << "bll";
    double *temp = new double[size]; 
    for(int i=0;i<size;i++) {
      temp[i]=data[i];
    }
    int *pivot = new int[length];
//    double *work = new double[3*size];
//    int lwork = 3*size;
    int nrhs = 1;
    int info;
    int lda=length;
    char trans = 'N';
    dgetrf_(&length,&height,temp,&lda,pivot,&info); 
    dgetrs_(&trans,&length,&nrhs,temp,&lda,pivot,b.data,&lda,&info);
    delete[] pivot;
    delete[] temp;
    return 1;
#endif
}

template <class T>
int MatrixMTL<T>::solveFromLU(VectorMTL<T> &idx, VectorMTL<T> &b) {
  double temp;
  int index;
  T sum;
  int offset=0;
  for (int i=0;i<height;i++) {
    if ((index=int(idx.data[i]))!=i) {
      b.switchElements(i,index);
    }
    sum=b.data[i];
    for(int j=0;j<=i-1;j++)
      sum-=data[offset+j]*b.data[j];
    b.data[i]=sum;
    offset+=length;
  }

  offset = (height-1)*length;
  for (int i=height-1;i>=0;i--) {
    sum=b.data[i];
    for (int j=i+1;j<height;j++) {
      sum -= data[offset+j]*b.data[j];
    }
    if((temp=data[offset+i])==0) {
      cout << "Singularity problem\n";
      return 0;
    }
    b.data[i]=sum/temp;
    offset-=length;
  }
  return 1;
}

template <class T>
MatrixMTL<T> MatrixMTL<T>::dotFactorial(const int input) {
   MatrixMTL<T> output(height,length);

  output=*this;

  for (int i=1; i<input; i++) {
    output = output.dot(output);
  }

  return output;
}

template <class T>
MatrixMTL<T>::~MatrixMTL() {
  delete[] data;
}

template <class T>
MatrixMTL<T>::MatrixMTL(const TriDiagMatrixMTL<T> &input) {
    height=input.size;
    length=height;
    size=height*length;

    data = new T[size];
    data[0]=input.diag[0];
    data[1]=input.upper[0];
    for(int j=2;j<height;j++) {
      data[j]=0;
    }

    for(int i=1;i<height-1;i++) {
      int offset=i*height;
      for(int j=0;j<i-1;j++) {
        data[offset+j]=0;
      }
      data[offset+i-1]=input.lower[i-1];
      data[offset+i]=input.diag[i];
      data[offset+i+1]=input.upper[i];
      for(int j=i+2;j<height;j++) {
        data[offset+j]=0;
      }
    }
    int offset=(height-1)*height;
    for(int j=0;j<height-2;j++) {
      data[offset+j]=0;
    }
    data[offset+height-2]=input.lower[height-2];
    data[offset+height-1]=input.diag[height-1];
}

template <typename T> 
T norm(const MatrixMTL<T> &input) {
  T output = 0; 
  int size = input.size;
  T temp;

  for (int i=0; i<size; i++) {
    if ((temp=fabs(input.data[i]))>output) {
      output = temp;
    }
  }
  return output;
}

template <typename T>
T diffNorm(const MatrixMTL<T> &input1, const MatrixMTL<T> &input2) {
  T norm = fabs(input1.data[0]-input2.data[0]);
  T temp;

  for(int i=1;i<input1.size;i++) {
    if((temp=fabs(input1.data[i]-input2.data[i]))>norm) norm = temp;
  }

  return norm;
}


template <typename T>
ostream& operator<<(ostream &output, const MatrixMTL<T> &p) {
  for (int j=0; j<p.height; j++) {
    for (int i=0; i<p.length; i++) {
      output << p.data[j*p.length+i] << "\t";
    }
    output << endl;
  }
   return output;
}


template <typename T> 
inline const MatrixMTL<T> operator *(const MatrixMTL<T> &self, const MatrixMTL<T> &input) {
    return MatrixMTL<T>(self)*=input;
}	

template <typename T> 
inline const MatrixMTL<T> operator *(const MatrixMTL<T> &self, const T input) {
    return MatrixMTL<T>(self)*=input;
}

template <typename T> 
inline const MatrixMTL<T> operator *(const T input, const MatrixMTL<T> &self) {
    return MatrixMTL<T>(self)*=input;
}	

template <typename T> 
inline const MatrixMTL<T> operator +(const MatrixMTL<T> &self, const MatrixMTL<T> &input) {
    return MatrixMTL<T>(self)+=input;
}	

template <typename T> 
inline const MatrixMTL<T> operator +(const T input, const MatrixMTL<T> &self) {
    return MatrixMTL<T>(self)+=input;
}	

template <typename T> 
inline const MatrixMTL<T> operator +(const MatrixMTL<T> &self, const T input) {
    return MatrixMTL<T>(self)+=input;
}

template <typename T> 
inline const MatrixMTL<T> operator /(const MatrixMTL<T> &self, const MatrixMTL<T> &input) {
    return MatrixMTL<T>(self)/=input;
}	

template <typename T> 
inline const MatrixMTL<T> operator /(const MatrixMTL<T> &self, const T input) {
    return MatrixMTL<T>(self)/=input;
}

template <typename T> 
inline const MatrixMTL<T> operator /(const T lhs, const MatrixMTL<T> &input) {
    MatrixMTL<T> output(input.size);

    for (int i=0; i<input.size; i++) {
      output.data[i]-=lhs/input.data[i];
    }

    return output;
}	

template <typename T> 
inline const MatrixMTL<T> operator -(const MatrixMTL<T> &self, const MatrixMTL<T> &input) {
    return MatrixMTL<T>(self)-=input;
}	

template <typename T> 
inline const MatrixMTL<T> operator -(const MatrixMTL<T> &self, const T input) {
    return MatrixMTL<T>(self)-=input;
}

template <typename T> 
inline const MatrixMTL<T> operator -(const T input, const MatrixMTL<T> &self) {
    return -MatrixMTL<T>(self)+=input;
}	

template <typename T> 
int MatrixMTL<T>::statDistr(VectorMTL<T> &expfreq) {
  int offset1;
  int offset2;
  T temp;
  for(int i=0; i<height-1; i++) {
    offset1 = i*length;
    offset2 = height*(i+1)+i;
    data[offset1+i]--;
    for(int j=i+1; j<length; j++) {
      temp = data[offset1+j];
      data[offset1+j]=data[offset2];
      data[offset2]=temp;
//        cout << offset1+j << " " << offset2 <<endl;
//        getchar();
       offset2+=height;
    }
  }

  offset1 = height*(length-1);
  for(int i=0; i<length; i++) {
    data[offset1+i]=1.0;
  }

  expfreq=0.0;
  expfreq[length-1]=1.0;

  if(!(solve(expfreq))) {
    cout << "Cannot compute invariant distribution\n";
    exit(0);
  }
}

template <typename T>
MatrixMTL<T> mSqrt(const MatrixMTL<T> &input) {
    MatrixMTL<T> output(input.height, input.length);

    for (int i=0; i<input.size; i++) {
      output.data[i]=sqrt(input.data[i]);
    }

    return output;
}

template <typename T>
MatrixMTL<T> mAbs(const MatrixMTL<T> &input) {
    MatrixMTL<T> output(input.height, input.length);

    for (int i=0; i<input.size; i++) {
      output.data[i]=fabs(input.data[i]);
    }

    return output;
}


}
#endif
