/***************************************************************************
 *   Copyright (C) 2006 by Jeziorski, Weintraub, Benkard and Van Roy       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef VECTORMTL_H
#define VECTORMTL_H

#include<iostream>
#include<math.h>
#include<rangenlib.h>
#include<matrixMTL.h>
#include<tridiagmatrix.h>
#include<sparsematrixMTL.h>


using namespace std;

namespace MTL {

// Forward delarations
template <class T> class VectorMTL;
template <class T> class MatrixMTL;
template <class T> class TriDiagMatrixMTL;
template <class T> class SparseMatrixMTL;

template <typename T> inline const VectorMTL<T> operator+(const VectorMTL<T>&, const VectorMTL<T>&);
template <typename T> inline const VectorMTL<T> operator+(const VectorMTL<T>&, const T);
template <typename T> inline const VectorMTL<T> operator+(const T, const VectorMTL<T>&);
template <typename T> inline const VectorMTL<T> operator-(const VectorMTL<T>&, const VectorMTL<T>&);
template <typename T> inline const VectorMTL<T> operator-(const VectorMTL<T>&, const T);
template <typename T> inline const VectorMTL<T> operator-(const T, const VectorMTL<T>&);
template <typename T> inline const VectorMTL<T> operator*(const VectorMTL<T>&, const VectorMTL<T>&);
template <typename T> inline const VectorMTL<T> operator*(const VectorMTL<T>&, const T);
template <typename T> inline const VectorMTL<T> operator*(const T, const VectorMTL<T>&);
template <typename T> inline const VectorMTL<T> operator/(const VectorMTL<T>&, const VectorMTL<T>&);
template <typename T> inline const VectorMTL<T> operator/(const VectorMTL<T>&, const T);
template <typename T> inline const VectorMTL<T> operator/(const T, const VectorMTL<T>&);
template <typename T> std::ostream& operator<<(std::ostream&, const VectorMTL<T>&);

template <typename T> T sum(const VectorMTL<T> &);
template <typename T> T mean(const VectorMTL<T> &);
template <typename T> T var(const VectorMTL<T> &);
template <typename T> T max(const VectorMTL<T> &);
template <typename T> T min(const VectorMTL<T> &);
template <typename T> T diffNorm(const VectorMTL<T> &input1, const VectorMTL<T> &input2);
template <typename T> VectorMTL<T> cumSum(const VectorMTL<T> &);
template <typename T> VectorMTL<T> flip(const VectorMTL<T> &);

template <typename T> VectorMTL<T> vExp(const VectorMTL<T> &);
template <typename T> VectorMTL<T> vLn(const VectorMTL<T> &);
template <typename T> VectorMTL<T> vSqrt(const VectorMTL<T> &);
template <typename T> VectorMTL<T> vAbs(const VectorMTL<T> &);
template <typename T> VectorMTL<T> vFact(const VectorMTL<T> &);

template <typename T> T norm(const VectorMTL<T> &);
template <typename T> VectorMTL<T> positive(const VectorMTL<T> &);

template <typename T> int genDistr(RanGenLib &rnd, const VectorMTL<T> &input);

/**
	@author Przemyslaw Jeziorski <przemekj@stanford.edu>
        \brief VectorMTL Class implements basic vector operations

        My Template Library Vector Class implements basic vector oparations
        like addition and multiplication.
        It also has methods to do more complicated operations like 
        maximum, minimum or cumulated sum.
*/

template <class T> 
class VectorMTL {
  friend class MatrixMTL<T>;
  friend class TriDiagMatrixMTL<T>;
  friend class SparseMatrixMTL<T>;
  /**
	Overloaded IOStream operator to display a Vector
  */
  friend std::ostream& operator<< <T>(std::ostream &, const VectorMTL<T> &);

  /**
	Addition of two vectors
        \param VectorMTL class, first agument in addition
        \param VectorMTL class, second argument in addition
  */
  friend const VectorMTL<T> operator+<T>(const VectorMTL<T>&, const VectorMTL<T>&);

  /**
	Addition of number to the vector
        \param Template type, first agument in addition
        \param VectorMTL class, second argument in addition
  */
  friend const VectorMTL<T> operator+<T>(const VectorMTL<T>&, const T);

  /**
	Addition of vector to the number
        \param VectorMTL class, first agument in addition
        \param Template type, second argument in addition
  */
  friend const VectorMTL<T> operator+<T>(const T, const VectorMTL<T>&);

  /**
	Subtraction of two vectors A-B
        \param VectorMTL class, A
        \param VectorMTL class, B
  */
  friend const VectorMTL<T> operator-<T>(const VectorMTL<T>&, const VectorMTL<T>&);

  /**
	Subtraction of a scalar (A) from a vector (B) A-B
        \param VectorMTL class, A
        \param Template type, B
  */
  friend const VectorMTL<T> operator-<T>(const VectorMTL<T>&, const T);

  /**
	Subtraction of a vector (A) from a scalar (B) A-B
        \param Template type, A
        \param VectorMTL class, B
  */
  friend const VectorMTL<T> operator-<T>(const T, const VectorMTL<T>&);

  /**
	Multiplication of two vectors A*B, elementwise
        \param VectorMTL class, A
        \param VectorMTL class, B
  */
  friend const VectorMTL<T> operator*<T>(const VectorMTL<T>&, const VectorMTL<T>&);

  /**
	Multiplication of a vector (A) by a scalar (B) A*B, elementwise
        \param VectorMTL class, A
        \param Template type, B
  */
  friend const VectorMTL<T> operator*<T>(const VectorMTL<T>&, const T);

  /**
	Multiplication of a scalar (A) by a vector (B) A*B, elementwise
        \param Template type, A
        \param VectorMTL class, B
  */
  friend const VectorMTL<T> operator*<T>(const T, const VectorMTL<T>&);

  /**
	Division of two vectors A/B, elementwise
        \param VectorMTL class, A
        \param VectorMTL class, B
  */
  friend const VectorMTL<T> operator/<T>(const VectorMTL<T>&, const VectorMTL<T>&);

  /**
	Division of a vector (A) by a scalar (B) A/B, elementwise
        \param VectorMTL class, A
        \param Template type, B
  */
  friend const VectorMTL<T> operator/<T>(const VectorMTL<T>&, const T);

  /**
	Division of a scalar (A) by a vector (B) A/B, elementwise
        \param Template type, A
        \param VectorMTL class, B
  */
  friend const VectorMTL<T> operator/<T>(const T, const VectorMTL<T>&);

  /**
	Maximum norm of a vector
        \param VectorMTL
  */
  friend T norm<T>(const VectorMTL<T> &);

  /**
	Norm of the difference of two vectors
	\param input1
        \param input2
 */
  friend T diffNorm<T>(const VectorMTL<T> &input1, const VectorMTL<T> &input2);

  /**
	Sum of the elements
        \param VectorMTL
  */
  friend T sum<T>(const VectorMTL<T> &);

  /**
	Mean of the elements
        \param VectorMTL
  */
  friend T mean<T>(const VectorMTL<T> &);

  /**
	Variance of the elements
        \param VectorMTL
  */
  friend T var<T>(const VectorMTL<T> &);

  /**
	Maximal element
        \param VectorMTL
  */
  friend T max<T>(const VectorMTL<T> &);

  /**
	Minimal element
        \param VectorMTL
  */
  friend T min<T>(const VectorMTL<T> &);

  /**
	Cumulative sum of elements
        \param VectorMTL
  */
  friend VectorMTL<T> cumSum<T>(const VectorMTL<T> &);

  /**
	Flip a vector
        \param VectorMTL
  */
  friend VectorMTL<T> flip<T>(const VectorMTL<T> &);

  /**
	Exponential function of a vector, elementwise
        \param VectorMTL
  */
  friend VectorMTL<T> vExp<T>(const VectorMTL<T> &);

  /**
	Natural log of a vector, elementwise
        \param VectorMTL
  */
  friend VectorMTL<T> vLn<T>(const VectorMTL<T> &);

  /**
	Square root of a vector, elementwise
        \param VectorMTL
  */
  friend VectorMTL<T> vSqrt<T>(const VectorMTL<T> &);

  /**
	Absolute value of a vector, elementwise
        \param VectorMTL
  */
  friend VectorMTL<T> vAbs<T>(const VectorMTL<T> &);

  /**
	Factorial of a vector, elementwise
        \param VectorMTL
  */
  friend VectorMTL<T> vFact<T>(const VectorMTL<T> &);

  private:
  int size;

  public:
  VectorMTL<T> &subVectorRef(int index, int length);

  /**
	Set the size of a vector.
        \param Integer
  */
  void setSize(int sizeInit);

  /**
	Set the data array
        \param Array of template type. Has to be the same size as a vector
  */
  void setData(T *dataInit); 
  T *data;

  VectorMTL<T> &operator=(T);
  VectorMTL<T> &operator=(const VectorMTL<T>&);


  /*** constuctors ***/

  /**
    Empty constructor
  */
  VectorMTL() {
    data = NULL;
  }

  /**
	Contructor
        \param Size of the array
        \param Array of template type
  */
  VectorMTL(int sizeInit,T *dataInit) {
    data = dataInit;
    size = sizeInit;
  }

  /**
	Initialize the vector. Should be called only of the vector was costructed with an empty constructor
        \param Array of template type
        \param Size of the array
  */
  void init(int sizeInput) {
    size=sizeInput;
    data = new T[size];
  }


  /**
	Contructor that initializes the empty vector.
        \param Size of the vector
  */
  VectorMTL(int sizeInput) {
    size = sizeInput;
    data = new T[size];
  }

  /**
	Copy Contructor
        \param copy
  */
  VectorMTL(const VectorMTL<T> &copy) {
    size = copy.size;
    data = new T[size];
    for (int i=0; i<size; i++) {
      data[i]=copy.data[i];
    }
  }

  /*** operators ***/

  inline VectorMTL &operator +=(const VectorMTL &input) {
    for (int i=0; i<size; i++) {
     data[i]+=input.data[i];
    }

    return *this;
  }

  inline VectorMTL &operator +=(const T input) {
    for (int i=0; i<size; i++) {
      data[i]+=input;
    }

    return *this;
  }

  inline VectorMTL &operator *=(const VectorMTL<T> &input) {
    for (int i=0; i<size; i++) {
      data[i]*=input.data[i];
    }

    return *this;
  }

//   inline VectorMTL &operator ^(const int p) {
//     for (int i=0; i<size; i++) {
//       double temp=data[i];
//       for (int j=1; j<p; j++) {
//         temp*=data[i];
//       }
//       
//     }
// 
//     return *this;
//   }

  inline VectorMTL &operator *=(const T input) {
    for (int i=0; i<size; i++) {
      data[i]*=input;
    }

    return *this;
  }

  inline VectorMTL &operator /=(const VectorMTL<T> &input) {
    for (int i=0; i<size; i++) {
      data[i]/=input.data[i];
    }

    return *this;
  }

  inline VectorMTL &operator /=(const T input) {
    for (int i=0; i<size; i++) {
      data[i]/=input;
    }

    return *this;
  }

  inline VectorMTL &operator -=(const VectorMTL<T> &input) {
    for (int i=0; i<size; i++) {
      data[i]-=input.data[i];
    }

    return *this;
  }

  inline VectorMTL &operator -=(const T input) {
    for (int i=0; i<size; i++) {
      data[i]-=input;
    }

    return *this;
  }


  inline T &operator[](const int input) {
    return data[input];
  }

  VectorMTL<T> operator-() {
    VectorMTL<T> output(size);

    for (int i=0; i<size; i++) {
      output.data[i]=-data[i];
    }

    return output;
  }

  /**
	Dot product
        \param VectorMTL
  */
  T dot(const VectorMTL<T> &input);

  /**
	Create base vector
        \param Integer, position of 1
  */
  void base(const int input);

  /**
	Returns size of the vector
  */
  int getSize();

  /**
	Returns pointer to internal array
  */
  T *getData() {
    return data;
  }

  /**
	Switch emelents
        \param Element 1 index
        \param Element 2 index
  */
  void switchElements(const int x, const int y);

  T norm();

  void addElement(VectorMTL<T> input);
  void addElement(T input);
  void newSize(int input, T init);
  ~VectorMTL();

  /**
	Generate vector of random draws from the simple pdf.
        Output vector A is such that A[i] is number of times that number
        'i' was drawn.
        \param RanGenLib class, random number generator
        \param VectorMTL, pdf
        \param Integer, number of draws
  */
  void genFixed(RanGenLib &rnd, VectorMTL<T> &input, int number) {
    VectorMTL<T> cumulate = cumSum(input);

    *this=0;
    for(int i=0;i<number;i++) {
      double a = rnd.genUniform();
      for(int j=0;j<size;j++) {
        if(a<cumulate.data[j]) {
          data[j]++;
          break;
        }
      }
    }
  }

  /** 
	Generates draw from the distribution
	\param input Disturibution
  */
  friend int genDistr<T>(RanGenLib &rnd, const VectorMTL<T> &input);

  /**
        Simulate a Markov chain. 
        Output vector A is such that A[i] is number of times that state
        'i' was drawn.
        \param RanGenLib class, random number generator
        \param MatrixMTL, transition matrix
        \param VectorMTL, each element 'i' of the vector specifies number of draws 
        that should be done with 'i' as a starting state
  */
  void genMarkov(RanGenLib &rnd, MatrixMTL<T> &probs, VectorMTL<T> number) {
     for(int i=0;i<size;i++) {
       data[i]=0;
     }

     for(int fromState=0; fromState<number.size; fromState++) {
       VectorMTL<T> pdf = probs.getRow(fromState);
       VectorMTL<T> cumulate = cumSum(pdf);
       for(int i=0;i<number.data[fromState];i++) {
         double a = rnd.genUniform();
         for(int j=0;j<size;j++) {
           if(a<cumulate.data[j]) {
             data[j]++;
             break;
           }
         }
       }
     }
   }

  /**
        Tridiagonal version of simulating a Markov chain. 
        Output vector A is such that A[i] is number of times that state
        'i' was drawn.
        \param RanGenLib class, random number generator
        \param TriDiagMatrixMTL, transition matrix
        \param VectorMTL, each element 'i' of the vector specifies number of draws 
        that should be done with 'i' as a starting state
  */
  void genMarkov(RanGenLib &rnd, TriDiagMatrixMTL<T> &probs, VectorMTL<T> number) {
     for(int i=0;i<size;i++) {
       data[i]=0;
     }

     for(int fromState=0; fromState<number.size; fromState++) {
       VectorMTL<T> pdf = probs.getRow(fromState);
       VectorMTL<T> cumulate = cumSum(pdf);
       for(int i=0;i<number.data[fromState];i++) {
         double a = rnd.genUniform();
         for(int j=0;j<size;j++) {
           if(a<cumulate.data[j]) {
             data[j]++;
             break;
           }
         }
       }
     }
   }

  /**
        Sparse version of simulating a Markov chain. 
        Output vector A is such that A[i] is number of times that state
        'i' was drawn.
        \param RanGenLib class, random number generator
        \param TriDiagMatrixMTL, transition matrix
        \param VectorMTL, each element 'i' of the vector specifies number of draws 
        that should be done with 'i' as a starting state
  */
  void genMarkov(RanGenLib &rnd, SparseMatrixMTL<T> &probs, VectorMTL<T> number) {
     for(int i=0;i<size;i++) {
       data[i]=0;
     }

     for(int fromState=0; fromState<number.size; fromState++) {
       VectorMTL<T> pdf = probs.getRow(fromState);
       VectorMTL<T> cumulate = cumSum(pdf);
       for(int i=0;i<number.data[fromState];i++) {
         double a = rnd.genUniform();
         for(int j=0;j<size;j++) {
           if(a<cumulate.data[j]) {
             data[j]++;
             break;
           }
         }
       }
     }
   }


  /**
        Generate vector of Poisson random variables
        \param RanGenLib class, random number generator
        \param VectorMTL, means of subsequent draws
  */
  void genPoisson(RanGenLib &rnd, VectorMTL<T> &lambda) {
    for(int i=0;i<size;i++) {
      data[i]=rnd.genPoisson(lambda.data[i]);
    }
  }

  /**
        Get the subvector
        \param Integer, staring from
        \param Integer, last element
  */
  VectorMTL<T> subVector(const int start, const int end) {
    int newSize = end-start+1;
    VectorMTL<T> output(newSize);
    for(int i=start;i<=end;i++) {
      output.data[i-start] = data[i];
    }
    return output;
  }
};

template <class T>
VectorMTL<T>::~VectorMTL() {
  if (data!=NULL) {
    delete[] data;
  }
}

template <class T>
void VectorMTL<T>::setData(T *dataInit) {
  data = dataInit;
}

template <class T>
void VectorMTL<T>::setSize(int sizeInit) {
  size = sizeInit;
}

template <class T>
VectorMTL<T> &VectorMTL<T>::subVectorRef(int index, int length) {
  VectorMTL<T> *output = new VectorMTL<T>;
  output->setSize(length);
  output->setData(data);

  return *output;
}

template <class T>
T VectorMTL<T>::dot(const VectorMTL<T> &input) {
  T output=data[0]*input.data[0];

  for (int i=1; i<size; i++) {
	output+=data[i]*input.data[i];
  }

  return output;
}

/****************/
/* basis vector */
/****************/
template <class T>
void VectorMTL<T>::base(const int input) {
  for(int i=0; i<size; i++) {
   data[i]=0;
  }
  data[input]=1;
}


template <class T>
int VectorMTL<T>::getSize() {
  return size;
}

template <class T>
void VectorMTL<T>::switchElements(const int x, const int y) {
  T temp;
  temp=data[x];
  data[x]=data[y];
  data[y]=temp;
}

template <class T>
T VectorMTL<T>::norm() {
  T output=0, temp;

  for (int i=0; i<size; i++) {
    if ( (temp=fabs(data[i]))>output ) {
      output = temp;
    }
  }

  return output;
}

template <typename T>
VectorMTL<T> positive(const VectorMTL<T> &input) {
  VectorMTL<T> output(input.size);
  for(int i=0; i<input.size; i++) {
    output.data[i] = (input.data[i]<0) ? 0 : input.data[i];
  }

  return output;
} 

template <typename T> 
T norm(const VectorMTL<T> &input) {
  T output=0;
  int size = input.size;
  T temp;

  for (int i=0; i<size; i++) {
    temp = fabs(input.data[i]);
    if (temp>output) {
      output = temp;
    }
  }
  return output;
}

template <class T>
void VectorMTL<T>::addElement(T input) {
  T *temp = data; 

  data = new T[size+1];
  for (int i=0;i<size; i++) {
    data[i]=temp[i];
  }
  data[size]=input;
  size++;

  delete[] temp;
}

template <class T>
void VectorMTL<T>::addElement(VectorMTL<T> input) {
  int oldSize = size;
  T *temp = data; 

  size += input.getSize();

  data = new T[size];


  for (int i=0;i<oldSize; i++) {
    data[i]=temp[i];
  }
  for (int i=0;i<input.getSize();i++) {
    data[oldSize+i]=input[i];
  }

  delete[] temp;
}

template <class T> 
void VectorMTL<T>::newSize(int input, T init) {
  T *temp = data; 
  int oldSize = size;
  int add = input-oldSize;

  if(add<0) {
    size = input;
    data = new T[size];
    for(int i=0;i<input;i++) {
      data[i]=temp[i];
    }
    delete[] temp;
  } else {
    size = input;
    data = new T[size];
    for (int i=0;i<oldSize; i++) {
      data[i]=temp[i];
    }
    for (int i=0;i<add;i++) {
      data[oldSize+i]=init;
    }
    delete[] temp;
  }
}

template <class T> 
VectorMTL<T> &VectorMTL<T>::operator=(const T input) {
    if(data==NULL) {
      data = new T[1];
      size = 1;
    }
    for (int i=0; i<size; i++) {
      data[i]=input;
    }

    return *this;
}	

template <class T> 
VectorMTL<T> &VectorMTL<T>::operator=(const VectorMTL<T> &input) {
    if(input.size!=size) {
//      cout << "shit" <<endl;
//      exit(1);
      if(data==NULL) {
        delete[] data;
      }
      size = input.size;
      data = new T[size];
    }
    for (int i=0; i<size; i++) {
      data[i]=input.data[i];
    }
    return *this;
}

template <typename T> std::ostream& operator<< (std::ostream &output, const VectorMTL<T> &p) {
  for (int i=0; i<p.size; i++) {
   output << p.data[i] << "\t\t";
  }

  output << endl;

  return output;
}

template <typename T> 
inline const VectorMTL<T> operator *(const VectorMTL<T> &self, const VectorMTL<T> &input) {
    return VectorMTL<T>(self)*=input;
}	

template <typename T> 
inline const VectorMTL<T> operator *(const VectorMTL<T> &self, const T input) {
    return VectorMTL<T>(self)*=input;
}

template <typename T> 
inline const VectorMTL<T> operator *(const T input, const VectorMTL<T> &self) {
    return VectorMTL<T>(self)*=input;
}	

template <typename T> 
inline const VectorMTL<T> operator +(const VectorMTL<T> &self, const VectorMTL<T> &input) {
    return VectorMTL<T>(self)+=input;
}	

template <typename T> 
inline const VectorMTL<T> operator +(const T input, const VectorMTL<T> &self) {
    return VectorMTL<T>(self)+=input;
}	

template <typename T> 
inline const VectorMTL<T> operator +(const VectorMTL<T> &self, const T input) {
    return VectorMTL<T>(self)+=input;
}

template <typename T> 
inline const VectorMTL<T> operator /(const VectorMTL<T> &self, const VectorMTL<T> &input) {
    return VectorMTL<T>(self)/=input;
}	

template <typename T> 
inline const VectorMTL<T> operator /(const VectorMTL<T> &self, const T input) {
    return VectorMTL<T>(self)/=input;
}

template <typename T> 
inline const VectorMTL<T> operator /(const T lhs, const VectorMTL<T> &input) {
    VectorMTL<T> output(input.size);

    for (int i=0; i<input.size; i++) {
      output.data[i]=lhs/input.data[i];
    }

    return output;
}	

template <typename T> 
inline const VectorMTL<T> operator -(const VectorMTL<T> &self, const VectorMTL<T> &input) {
    return VectorMTL<T>(self)-=input;
}	

template <typename T> 
inline const VectorMTL<T> operator -(const VectorMTL<T> &self, const T input) {
    return VectorMTL<T>(self)-=input;
}

template <typename T> 
inline const VectorMTL<T> operator -(const T input, const VectorMTL<T> &self) {
    return -VectorMTL<T>(self)+=input;
}	

template <typename T> 
T max(const VectorMTL<T> &input) {
    T output = input.data[0];
    T size = input.size;
    for(int i=1;i<size;i++) 
      if(input.data[i]>output) 
        output=input.data[i];
    return output;
} 

template <typename T> 
T min(const VectorMTL<T> &input) {
    T output = input.data[0];
    T size = input.size;
    for(int i=1;i<size;i++) 
      if(input.data[i]<output) 
        output=input.data[i];
    return output;
} 

template <typename T> 
VectorMTL<T> vSqrt(const VectorMTL<T> &input) {
    VectorMTL<T> output(input.size);

    for (int i=0; i<input.size; i++) {
      output.data[i]=sqrt(input.data[i]);
    }

    return output;
}

template <typename T> 
VectorMTL<T> vAbs(const VectorMTL<T> &input) {
    VectorMTL<T> output(input.size);

    for (int i=0; i<input.size; i++) {
      output.data[i]=fabs(input.data[i]);
    }

    return output;
}

template <typename T> 
VectorMTL<T> vExp(const VectorMTL<T> &input) {
   VectorMTL<T> output(input.size);


    for (int i=0; i<input.size; i++) {
      T temp = input.data[i];
      output.data[i]=exp(temp);
    }

    return output;
}

template <typename T> 
VectorMTL<T> cumSum(const VectorMTL<T> &input) {
    int sizeOut = input.size;
    VectorMTL<T> output(sizeOut);
    T sum=0;
    for(int i=0;i<sizeOut;i++) {
      sum+=input.data[i];
      output.data[i]=sum;
    }

    return output;
}

template <typename T> 
VectorMTL<T> flip(const VectorMTL<T> &input) {
    int sizeOut = input.size;
    VectorMTL<T> output(sizeOut);

    for(int i=0;i<sizeOut;i++) {
      output.data[i]=input.data[sizeOut-1-i];
    }
    return output;
}

template <typename T>
VectorMTL<T> vLn(const VectorMTL<T> &input) {
  VectorMTL<T> output(input.size);

  for (int i=0; i<input.size; i++) {
	output.data[i]=log(input.data[i]);
  }

  return output;
}

template <typename T>
T sum(const VectorMTL<T> &input) {
  T output=input.data[0];

  for (int i=1; i<input.size; i++) {
    output+=input.data[i];
  }

  return output;
}

template <typename T>
T mean(const VectorMTL<T> &input) {
  return sum(input)/input.size;
}

template <typename T>
T var(const VectorMTL<T> &input) {
  T temp=input.data[1]*input.data[1];
  for (int i=1; i<input.size; i++) {
    temp+=input.data[i]*input.data[i];
  }

 return (sum(input)^2-temp)/input.size; 
}

template <typename T>
T diffNorm(const VectorMTL<T> &input1, const VectorMTL<T> &input2) {
  T norm = fabs(input1.data[0]-input2.data[0]);
  T temp;

  for(int i=1;i<input1.size;i++) {
    if((temp=fabs(input1.data[i]-input2.data[i]))>norm) norm = temp;
  }

  return norm;
}

template <typename T>
int genDistr(RanGenLib &rnd, const VectorMTL<T> &input) {
  VectorMTL<T> cumulate = cumSum(input);

  double a = rnd.genUniform();
  for(int j=0;j<input.size;j++) {
    if(a<cumulate.data[j]) {
      return j;
    }
  }
}

}
#endif
