/***************************************************************************
 *   Copyright (C) 2006 by Jeziorski, Weintraub, Benkard and Van Roy       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/** \file src/compoe/transitionMatrixTridiagonal.cpp 
  File contains sample implementation of the state transition matrix and
  transition matrix conditional on not exiting.
  This is file is used only of dense transition matrix is enabled in setup.h
  In this case one needs to specify all elements of the matrix
  It is the most flexible type of transition matrix. The cost is that no 
  sparsity structure can be specified that causes greater cost of inversion.
  In case of sparse or tridiagonal transition matrices one should use 
  appropriate *cpp file to speed up computations.

  Sample implementation:
   - State transition follows the equation
       $x_{t+1}=x_t+w(i_t,\zeta_{t+1})+\epsilon_{t+1}$
     where
       a) w(.) is the {-1,0,1} valued random variable that represents transitions in response to investment
       b) \epsilon_t is independent shock to the transition, distribution function of this RV is given 
          by appropriate transition(.) parameters in parameters.h or as arguments to the Matlab function.
	  For more information refer to the provided examples of configuration

  \param iota Investment function
  \param tranMatrix Output transition matrix
  \param contTranMatrix Output conditional transition matrix
  \param contProbVector Probabilities of not exiting
*/ 

#include<compoe_dom.h>

#ifdef ACW

void CompOE_dom::tranProbACWDom(MatrixMTL<double> &RHS, SparseMatrixMTL<double> &tranMatrix) {
  int actions=constants[6];
  VectorMTL<double> action_prob(actions);

  // Compute probabilitiles of all actions
  tranMatrix=0.0;
  for(int x=0;x<xmax;++x) {
    double denom=1;
    for(int a=0;a<actions-1;++a) {
      action_prob[a]=exp(RHS[x][a]/constants[7]);
      denom+=action_prob[a];
    }
    action_prob[actions-1]=1;
    action_prob/=denom;
    // Compute transitions
    for(int a=0;a<actions;++a) {
      tranMatrix[x*(xmax_dom)+(*acwTran)[a][x]]+=action_prob[a];
    }
  }
}

#endif
