/***************************************************************************
 *   Copyright (C) 2006 by Jeziorski, Weintraub, Benkard and Van Roy       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/** \file src/compoe/transitionMatrixTridiagonal.cpp 
  File contains sample implementation of the state transition matrix and
  transition matrix conditional on not exiting.
  This is file is used only of dense transition matrix is enabled in setup.h
  In this case one needs to specify all elements of the matrix
  It is the most flexible type of transition matrix. The cost is that no 
  sparsity structure can be specified that causes greater cost of inversion.
  In case of sparse or tridiagonal transition matrices one should use 
  appropriate *cpp file to speed up computations.

  Sample implementation:
   - State transition follows the equation
       $x_{t+1}=x_t+w(i_t,\zeta_{t+1})+\epsilon_{t+1}$
     where
       a) w(.) is the {-1,0,1} valued random variable that represents transitions in response to investment
       b) \epsilon_t is independent shock to the transition, distribution function of this RV is given 
          by appropriate transition(.) parameters in parameters.h or as arguments to the Matlab function.
	  For more information refer to the provided examples of configuration

  \param iota Investment function
  \param tranMatrix Output transition matrix
  \param contTranMatrix Output conditional transition matrix
  \param contProbVector Probabilities of not exiting
*/ 

#include<compoe.h>

void CompOE::tranProb(VectorMTL<double> &iota, MatrixMTL<double> &tranMatrixMTL, MatrixMTL<double> &contTranMartix, VectorMTL<double> &contProbVector) {
  VectorMTL<double> prup(xmax);
  VectorMTL<double> prsame(xmax);
  VectorMTL<double> prdown(xmax);
  VectorMTL<double> aiota(xmax);

  aiota=iota*transition[2];
  for (int i=0;i<xmax;i++) {
    prup[i]=(aiota[i]*(1-transition[1]))/(aiota[i]+1);                          // prob. going up one state
    prsame[i]=(aiota[i]*transition[1]+(1-transition[1]))/(aiota[i]+1);          // prob. stay in same state
    prdown[i]=transition[1]/(aiota[i]+1);                                       // prob. go down one state
  }

  tranMatrixMTL=0;
  contTranMartix=0;

  /*** Build tranMatrixMTL out of these rows */
  double allprob;
  int qstart=4;   // where q is in "transition"
  int farthestfall = int(transition[3])+1;
  int qlength = int(transition[3])*2+3;  // Goes from qstart to qstart+qlength-2
  for (int i=0;i<xmax;i++)
    for (int j=0;j<qlength;j++) {
      int k=i+j-farthestfall;        // so k goes from i-farthestfall to i+farthestrise
      k=(k<0)*0+(k>0)*(k<xmax)*k+(k>=xmax)*(xmax-1);
      allprob=prdown[i]*transition[qstart+j+2]+prsame[i]*transition[qstart+j+1]+prup[i]*transition[qstart+j];
      tranMatrixMTL[i][k]+=allprob;
      contTranMartix[i][k]+=allprob*contProbVector[i];
    }
}
