/***************************************************************************
 *   Copyright (C) 2006 by Jeziorski, Weintraub, Benkard and Van Roy       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/** \file src/compoe/transitionMatrixTridiagonal.cpp 
  Implementation of the transition matrix and
  transition matrix conditional on not exiting.
  Implement this file of you are using sparse matrix 
  solver i.e. you have #define TRANSITION_MATRIX 2 in setup.h
  Solver uses MUMPS to invert this matrix

  \param iota Investment function
  \param tranMatrix Output transition matrix
  \param contTranMatrix Output conditional transition matrix
  \param contProbVector Probabilities of not exiting
*/ 

#include<compoe.h>

void CompOE::tranProb(VectorMTL<double> &iota, SparseMatrixMTL<double> &tranMatrix, 
  SparseMatrixMTL<double> &contTranMatrix, VectorMTL<double> &contProbVector) {

  tranMatrix.bandMatrix(xmax,3);
  contTranMatrix.bandMatrix(xmax,3);

  VectorMTL<double> prup(xmax);
  VectorMTL<double> prsame(xmax);
  VectorMTL<double> prdown(xmax);
  VectorMTL<double> aiota(xmax);

  aiota=iota*transition[2];

  for (int i=0;i<xmax;i++) {
    prup[i]=(aiota[i]*(1-transition[1]))/(aiota[i]+1);           		// prob. going up one state
    prsame[i]=(aiota[i]*transition[1]+(1-transition[1]))/(aiota[i]+1);  	// prob. stay in same state 
    prdown[i]=transition[1]/(aiota[i]+1);					// prob. go down one state

//    prup[i]=MIN(prup[i]+0.01,0.99);
//    prdown[i]=MIN(MIN(prdown[i]+0.01,1-prup[i]),0.99);
//    prsame[i]=1-prup[i]-prdown[i];
  }
  prsame[xmax-1]=1-prdown[xmax-1];        // At state xmax can't go up
  prsame[0]=1-prup[0];                    // At state 1 can't go down

/* Using probs. to build matrix of transition probs. */
/* In the same time build matrix for contiunuing in the industry */

/* Create first row */
  tranMatrix[0]=prsame[0];
  tranMatrix[1]=prup[0];

  contTranMatrix[0]=prsame[0]*contProbVector[0];
  contTranMatrix[1]=prup[0]*contProbVector[0];

/* Create middle rows */
  for (int i=1, j=2; i<xmax-1; i++, j+=3) {
    tranMatrix[j]=prdown[i];
    tranMatrix[j+1]=prsame[i];
    tranMatrix[j+2]=prup[i];

    contTranMatrix[j]=prdown[i]*contProbVector[i];
    contTranMatrix[j+1]=prsame[i]*contProbVector[i];
    contTranMatrix[j+2]=prup[i]*contProbVector[i];
  }

/* Create last row */
  if (xmax>1) {
    int size = tranMatrix.getSize();
    tranMatrix[size-2]=prdown[xmax-1];
    tranMatrix[size-1]=prsame[xmax-1];

    contTranMatrix[size-2]=prdown[xmax-1]*contProbVector[xmax-1];
    contTranMatrix[size-1]=prsame[xmax-1]*contProbVector[xmax-1];
  }
}
