/***************************************************************************
 *   Copyright (C) 2006 by Jeziorski, Weintraub, Benkard and Van Roy       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef COMPOE_H
#define COMPOE_H

//#define OE_TEST


#include <iomanip>
#include<functionprofit.h>
#include<matrixMTL.h>
#include<matrix3d.h>
#include<tridiagmatrix.h>
#include<vectorMTL.h>
#include<sparsematrixMTL.h>
#include<setup.h>
#include<fstream>
#include<pthread.h>
#include<investmentCost.h>

#include<math.h>
typedef struct {
    VectorMTL<double> *state;
    int number;
    double output;
    VectorMTL<double> *p0;
    double *prof_fun;
    double *prof_fun_ipopt;
    int success;
  } ThreadStructure;

using namespace MTL;

/**
	@author Przemyslaw Jeziorski <przemekj@stanford.edu>
        \brief Computes basic version of Oblivious Equilibrium
*/

class CompOE{
  friend class CompOE_nonstat;
  friend class CompOE_aggr;
  friend class CompOE_dom;

private:
  ofstream *myfile;
  double lambda;
  int xmax;
  int subtract;
  int ignore;
  int success;
  int method;
  VectorMTL<double> *iota_global;
  VectorMTL<double> *rho_global;
  VectorMTL<double> *V_global;
  VectorMTL<double> *prof_global;
  VectorMTL<double> *prices_global; 
  VectorMTL<double> *tildes_global;
  VectorMTL<double> *prcont1;
  double bound1avg,bound1pre,bound1var;
  VectorMTL<double> *bound1por;
  VectorMTL<double> *bound2avg, *bound2pre, *bound2var, *bound2por;
  VectorMTL<double> *bound3avg, *bound3pre, *bound3var, *bound3por;
  VectorMTL<double> *bound4avg, *bound4pre, *bound4var, *bound4por;
  VectorMTL<double> *bound5avg, *bound5pre, *bound5var, *bound5por;
  VectorMTL<double> *firmsnumberavg, *firmsnumberpre, *firmsnumbervar, *firmsnumberpor;

  double condNumber;
  double prodsuravg,prodsurpre,prodsurvar;
  double conssuravg,conssurpre,conssurvar;
  double totalsuravg,totalsurpre,totalsurvar;
  double c1avg,c1pre,c1var;
  double c4avg,c4pre,c4var;
  double c2avg,c2pre,c2var;
  double c10avg,c10pre,c10var;
  double c20avg,c20pre,c20var;
  double hhi50avg, hhi50pre, hhi50var;
  double entrateavg,entratepre,entratevar;
  double exirateavg,exiratepre,exiratevar;
  double msexitavg,msexitpre,msexitvar;
  double invavg,invpre,invvar;
  int recomp;
  double *prof_fun;
  double *prof_fun_ipopt;
  double *transition;
  double *constants;
  double *init; 
  double *bounds;
  double *convtol;
  double *control;

  VectorMTL<double> *prodsuravg_path,*prodsurpre_path,*prodsurvar_path;
  VectorMTL<double> *conssuravg_path,*conssurpre_path,*conssurvar_path;
  VectorMTL<double> *totalsuravg_path,*totalsurpre_path,*totalsurvar_path;
  VectorMTL<double> *c1avg_path,*c1pre_path,*c1var_path;
  VectorMTL<double> *c4avg_path,*c4pre_path,*c4var_path;
  VectorMTL<double> *c2avg_path,*c2pre_path,*c2var_path;
  VectorMTL<double> *c10avg_path,*c10pre_path,*c10var_path;
  VectorMTL<double> *c20avg_path,*c20pre_path,*c20var_path;
  VectorMTL<double> *hhi50avg_path,*hhi50pre_path,*hhi50var_path;
  VectorMTL<double> *entrateavg_path,*entratepre_path,*entratevar_path;
  VectorMTL<double> *exirateavg_path,*exiratepre_path,*exiratevar_path;
  VectorMTL<double> *msexitavg_path,*msexitpre_path,*msexitvar_path;
  VectorMTL<double> *invavg_path,*invpre_path,*invvar_path;
  MatrixMTL<double> *firmsnumberavg_path, *firmsnumberpre_path, *firmsnumbervar_path, *firmsnumberpor_path;


  #ifdef ACW
    MatrixMTL<double> *action_profit;
    MatrixMTL<int> *acwTran;
    MatrixMTL3D<double> *RHSFringe;
  #endif


public:
  CompOE();

  /**
     Initialise the model parameters
     \param control_init Control parameters
     \param prof_fun_init Profit function parameters
     \param transition_init Transition matrix parameters
     \param constants_init Constants
     \param init_init Initialization parameters
     \param convtol_init Convergence parameters
  */

  inline void getStats(MatrixMTL<double> &stats) {
    stats[0][0]=conssuravg; stats[0][1]=conssurpre; stats[0][2]=conssurvar;
    stats[1][0]=prodsuravg; stats[1][1]=prodsurpre; stats[1][2]=prodsurvar;
    stats[2][0]=totalsuravg; stats[2][1]=totalsurpre; stats[2][2]=totalsurvar;

    stats[3][0]=c1avg; stats[3][1]=c1pre; stats[3][2]=c1var;
    stats[4][0]=c2avg; stats[4][1]=c2pre; stats[4][2]=c2var;
    stats[5][0]=c4avg; stats[5][1]=c4pre; stats[5][2]=c4var;
    stats[6][0]=c10avg; stats[6][1]=c10pre; stats[6][2]=c10var;
    stats[7][0]=c20avg; stats[7][1]=c20pre; stats[7][2]=c20var;

    stats[8][0]=hhi50avg; stats[8][1]=hhi50pre; stats[8][2]=hhi50var;

    //stats[9][0]=entrateavg; stats[9][1]=entratepre; stats[9][2]=entratevar;
    stats[9][0]=0; stats[9][1]=0; stats[9][2]=0;
    stats[10][0]=invavg; stats[10][1]=invpre; stats[10][2]=invvar;
  }

  void getStats_path(MatrixMTL<double> &stats_path, int M) {
    int j=0;

    for(int i=0;i<M;i++, j++) { 
      stats_path[j][0]=(*conssuravg_path)[i]; stats_path[j][1]=(*conssurpre_path)[i]; stats_path[j][2]=(*conssurvar_path)[i];
    }
    for(int i=0;i<M;i++, j++) { 
      stats_path[j][0]=(*prodsuravg_path)[i]; stats_path[j][1]=(*prodsurpre_path)[i]; stats_path[j][2]=(*prodsurvar_path)[i];
    }
    for(int i=0;i<M;i++, j++) { 
      stats_path[j][0]=(*totalsuravg_path)[i]; stats_path[j][1]=(*totalsurpre_path)[i]; stats_path[j][2]=(*totalsurvar_path)[i];
    }

    for(int i=0;i<M;i++, j++) {
      stats_path[j][0]=(*c1avg_path)[i]; stats_path[j][1]=(*c1pre_path)[i]; stats_path[j][2]=(*c1var_path)[i];
    }
    for(int i=0;i<M;i++, j++) {
      stats_path[j][0]=(*c2avg_path)[i]; stats_path[j][1]=(*c2pre_path)[i]; stats_path[j][2]=(*c2var_path)[i];
    }
    for(int i=0;i<M;i++, j++) {
      stats_path[j][0]=(*c4avg_path)[i]; stats_path[j][1]=(*c4pre_path)[i]; stats_path[j][2]=(*c4var_path)[i];
    }
    for(int i=0;i<M;i++, j++) {
      stats_path[j][0]=(*c10avg_path)[i]; stats_path[j][1]=(*c10pre_path)[i]; stats_path[j][2]=(*c10var_path)[i];
    }
    for(int i=0;i<M;i++, j++) {
      stats_path[j][0]=(*c20avg_path)[i]; stats_path[j][1]=(*c20pre_path)[i]; stats_path[j][2]=(*c20var_path)[i];
    }

    for(int i=0;i<M;i++, j++) {
      stats_path[j][0]=(*hhi50avg_path)[i]; stats_path[j][1]=(*hhi50pre_path)[i]; stats_path[j][2]=(*hhi50var_path)[i];
    }

    for(int i=0;i<M;i++, j++) {
      //stats_path[j][0]=(*entrateavg_path)[i]; stats_path[j][1]=(*entratepre_path)[i]; stats_path[j][2]=(*entratevar_path)[i];
      stats_path[j][0]=0; stats_path[j][1]=0; stats_path[j][2]=0;
    }
    for(int i=0;i<M;i++, j++) {
      stats_path[j][0]=(*invavg_path)[i]; stats_path[j][1]=(*invpre_path)[i]; stats_path[j][2]=(*invvar_path)[i];
    }
  }


  inline void initialize(double *control_init, double *prof_fun_init, double *prof_fun_ipopt_init,
    double *transition_init, double *constants_init, double *init_init, double *convtol_init) {

    control = control_init;
    prof_fun = prof_fun_init;
    prof_fun_ipopt = prof_fun_ipopt_init;
    transition = transition_init;
    constants = constants_init;
    init = init_init;
    convtol = convtol_init;


    // Set low sell-off value of constant number of firms
    if(control[2] != 0) 
      transition[0] = 1e-100;
  }

  inline void setMethod(int method_init) {
    method = method_init;
  }

  /**
     Set output file
     \param myfile_init Pointer to the ofstream
  */
  inline void setOutput(ofstream *myfileInit) {
    myfile = myfileInit;
  }

  /**
     Set initial Lambda
     \param lamda_init
  */
  inline void setLambda(double lambdaInit) {
    lambda = lambdaInit;
  }

  /**
     Set initial xmax
     \param xmaxInit
  */
  inline void setXmax(int xmaxInit) {
    xmax = xmaxInit;
  }

  /**
     Get condition number of the equilibrium transition matrix
  */
  inline double getCondNumber() {
    return condNumber;
  }

  /**
     Get the simulated mean of equilibrium producer surplus (need to compute bounds first)
  */
  inline double getProdsuravg() {
    return prodsuravg;
  }

  /**
     Get the simulated mean of equilibrium consumer surplus (need to compute bounds first)
  */
  inline double getConssuravg() {
    return conssuravg;
  }

  /**
     Get the simulated mean of equilibrium total surplus (need to compute bounds first)
  */
  inline double getTotalsuravg() {
    return totalsuravg;
  }

  /**
     Get the the simulated mean of equilibrium entry rate (need to compute bounds first)
  */
  inline double getEntrateavg() {
    return entrateavg;
  }

  /**
     Get the simulated mean of equilibrium exit rate (need to compute bounds first)
  */
  inline double getExirateavg() {
    return exirateavg;
  }

  /**
     Get the the simulated mean of equilibrium C1 (need to compute bounds first)
  */
  inline double getC1avg() {
    return c1avg;
  }

  /**
     Get the the simulated mean of equilibrium C4 (need to compute bounds first)
  */
  inline double getC4avg() {
    return c4avg;
  }

  /**
     Get the simulated mean of equilibrium C10 (need to compute bounds first)
  */
  inline double getC10avg() {
    return c10avg;
  }

  /**
     Get the simulated mean of equilibrium C20 (need to compute bounds first)
  */
  inline double getC20avg() {
    return c20avg;
  }

  /**
     Get the simulated mean of equilibrium market share of the exiting firm (need to compute bounds first)
  */
  inline double getMsexitavg() {
    return msexitavg;
  }

  /**
     Get the simulated variance of equilibrium producer surplus (need to compute bounds first)
  */
  inline double getProdsurvar() {
    return prodsurvar;
  }

  /**
     Get the simulated variance of equilibrium consumer surplus (need to compute bounds first)
  */
  inline double getConssurvar() {
    return conssurvar;
  }

  /**
     Get the simulated variance of equilibrium total surplus (need to compute bounds first)
  */
  inline double getTotalsurvar() {
    return totalsurvar;
  }

  /**
     Get the simulated variance of equilibrium entryrate surplus (need to compute bounds first)
  */
  inline double getEntratevar() {
    return entratevar;
  }

  /**
     Get the simulated variance of equilibrium exitrate surplus (need to compute bounds first)
  */
  inline double getExiratevar() {
    return exiratevar;
  }

  /**
     Get the simulated variance of equilibrium C1 (need to compute bounds first)
  */
  inline double getC1var() {
    return c1var;
  }

  /**
     Get the simulated variance of equilibrium C4 (need to compute bounds first)
  */
  inline double getC4var() {
    return c4var;
  }

  /**
     Get the simulated variance of equilibrium C10 (need to compute bounds first)
  */
  inline double getC10var() {
    return c10var;
  }

  /**
     Get the simulated variance of equilibrium C20 (need to compute bounds first)
  */
  inline double getC20var() {
    return c20var;
  }

  /**
     Get the simulated variance of equilibrium market share of exiting firm (need to compute bounds first)
  */
  inline double getMsexitvar() {
    return msexitvar;
  }

  inline void getFirmNumber(VectorMTL<double> &firmsnumberavgOut, VectorMTL<double> &firmsnumberpreOut,
          VectorMTL<double> &firmsnumbervarOut) {
    firmsnumberavgOut = *firmsnumberavg;
    firmsnumberpreOut = *firmsnumberpre;
    firmsnumbervarOut = *firmsnumbervar;
  }

  /**
     Compute continuation probability for a given exit strategy
     \param rho_local Exit strategy
     \param prcont_local Resulting contibuation probabilities
     \param condexp_local Conditional expectation of sell-off value upon exit
  */
  inline void continuationProb(VectorMTL<double> &rho_local, 
    VectorMTL<double> &prcont_local, VectorMTL<double> &condexp_local) {

    prcont_local = vExp((-rho_local)/transition[0]);

    for(int i=0;i<rho_local.getSize();i++) {
      if(prcont_local[i]<0) {
        prcont_local[i] = 1;
      } else if(prcont_local[i]>1) {
        prcont_local[i]=0;
      } else {
        prcont_local[i]=1-prcont_local[i];
      }
    }
    condexp_local = rho_local+transition[0];
  }

  inline void generatePoisson(RanGenLib &P,VectorMTL<double> &mean, VectorMTL<double> &output) {
    output.genPoisson(P,mean);
   /*int size = mean.getSize();
   for(int i=0;i<size; i++) {
     Poisson P(mean[i]);
     output[i]=P.Next();
   }*/
  }

  static void compAvgPre(VectorMTL<double> &xsum, VectorMTL<double> &xsqr, double n, 
    VectorMTL<double> &xavg, VectorMTL<double> &xpre, VectorMTL<double> &xvar) {
    xavg=xsum/n;
    xvar=(xsqr - (xavg*xavg*n))/(n-1);
    xpre=(vSqrt(xvar/n))/(vAbs(xavg))*2.326;
  }

  static void compAvgPre(MatrixMTL<double> &xsum, MatrixMTL<double> &xsqr, double n,
    MatrixMTL<double> &xavg, MatrixMTL<double> &xpre, MatrixMTL<double> &xvar) {
    xavg=xsum/n;
    xvar=(xsqr - (xavg*xavg*n))/(n-1);
    xpre=(mSqrt(xvar/n))/(mAbs(xavg))*2.326;
  }

  static void compAvgPre(double xsum, double xsqr, int n, 
    double &xavg, double &xpre, double &xvar) {
    xavg=xsum/n;
    xvar=(xsqr - (xavg*xavg*n))/(n-1);
    xpre=(sqrt(xvar/n))/(fabs(xavg))*2.326;

  }


  /**
     Get bound1
     \param bound1avgOut Average value of the bound
     \param bound1preOut relative precision with 98% confidence
     \param bound1varOut Variance
     \param bound1porOut Percentage of the value function
  */
  inline void getBound1(double &bound1avgOut, double &bound1preOut, 
    double &bound1varOut, VectorMTL<double> &bound1porOut) {

    bound1avgOut = bound1avg;
    bound1varOut = bound1var;
    bound1porOut = *bound1por;
    bound1preOut = bound1pre;
  }

  /**
     Get bound2
     \param bound2avgOut Average value of the bound
     \param bound2preOut relative precision with 98% confidence
     \param bound2varOut Variance
     \param bound2porOut Percentage of the value function
  */
  inline void getBound2(VectorMTL<double> &bound2avgOut, VectorMTL<double> &bound2preOut, 
    VectorMTL<double> &bound2varOut, VectorMTL<double> &bound2porOut) {

    bound2avgOut = *bound2avg;
    bound2varOut = *bound2var;
    bound2porOut = *bound2por;
    bound2preOut = *bound2pre;
  }

  /**
     Get bound3
     \param bound3avgOut Average value of the bound
     \param bound3preOut relative precision with 98% confidence
     \param bound3varOut Variance
     \param bound3porOut Percentage of the value function
  */
  inline void getBound3(VectorMTL<double> &bound3avgOut, VectorMTL<double> &bound3preOut, 
    VectorMTL<double> &bound3varOut, VectorMTL<double> &bound3porOut) {

    bound3avgOut = *bound3avg;
    bound3varOut = *bound3var;
    bound3porOut = *bound3por;
    bound3preOut = *bound3pre;
  }

  /**
     Get bound4
     \param bound4avgOut Average value of the bound
     \param bound4preOut relative precision with 98% confidence
     \param bound4varOut Variance
     \param bound4porOut Percentage of the value function
  */
  inline void getBound4(VectorMTL<double> &bound4avgOut, VectorMTL<double> &bound4preOut, 
    VectorMTL<double> &bound4varOut, VectorMTL<double> &bound4porOut) {

    bound4avgOut = *bound4avg;
    bound4varOut = *bound4var;
    bound4porOut = *bound4por;
    bound4preOut = *bound4pre;
  }

  /**
     Get bound5
     \param bound4avgOut Average value of the bound
     \param bound4preOut relative precision with 98% confidence
     \param bound4varOut Variance
     \param bound4porOut Percentage of the value function
  */
  inline void getBound5(VectorMTL<double> &bound5avgOut, VectorMTL<double> &bound5preOut, 
    VectorMTL<double> &bound5varOut, VectorMTL<double> &bound5porOut) {

    bound5avgOut = *bound5avg;
    bound5varOut = *bound5var;
    bound5porOut = *bound5por;
    bound5preOut = *bound5pre;
  }

  /**
	Get the pointer to the value function
  */
  inline VectorMTL<double> *getV() {
    return V_global;
  }

  /**
	Get the pointer to prices
  */
  inline VectorMTL<double> *getPrices() {
    return prices_global;
  }

  /**
	Get the probability of not exiting
  */
  inline VectorMTL<double> *getPrcont() {
    return prcont1;
  }

  /**
	Get Iota
  */
  inline VectorMTL<double> *getIota() {
    return iota_global;
  }

  /**
	Get Rho
  */
  inline VectorMTL<double> *getRho() {
    return rho_global;
  }

  /**
	Get expected state
  */
  inline VectorMTL<double> *getTildes() {
    return tildes_global;
  }

  /**
	Get entry rate
  */
  inline double getLambda() {
    return lambda;
  }

  /**
	Get xmax
  */
  inline int getXmax() {
    return xmax;
  }

  void tranProb(VectorMTL<double> &iota,MatrixMTL<double> &tranMatrixMTL, 
    MatrixMTL<double> &contTranMartix, VectorMTL<double> &contProbVector);

  void tranProb(VectorMTL<double> &iota, TriDiagMatrixMTL<double> &tranMatrixMTL, 
    TriDiagMatrixMTL<double> &contTranMartix, VectorMTL<double> &contProbVector);

  void tranProb(VectorMTL<double> &iota, SparseMatrixMTL<double> &tranMatrixMTL, 
    SparseMatrixMTL<double> &contTranMartix, VectorMTL<double> &contProbVector);

  void condToNoncondTranProb(TriDiagMatrixMTL<double> &, TriDiagMatrixMTL<double> &, VectorMTL<double> &);
  void condToNoncondTranProb(SparseMatrixMTL<double> &, SparseMatrixMTL<double> &, VectorMTL<double> &);
  void condToNoncondTranProb(MatrixMTL<double> &, MatrixMTL<double> &, VectorMTL<double> &);

  int compProf(VectorMTL<double> &, VectorMTL<double> &, VectorMTL<double> &);

  int compProfAndStats(VectorMTL<double> &state, VectorMTL<double> &prices, VectorMTL<double> 
    &pricesOutput, VectorMTL<double> &output, VectorMTL<double> &msOutput, 
    double &prodsurOutput, double &conssurOutput);

  int compProfAndStats_extra(VectorMTL<double> &state, VectorMTL<double> &prices, VectorMTL<double>
    &pricesOutput, VectorMTL<double> &output, VectorMTL<double> &msOutput,
    double &prodsurOutput, double &conssurOutput);

  int compProfAndStats_vector(VectorMTL<double> &state, VectorMTL<double> &prices, VectorMTL<double>
    &pricesOutput, VectorMTL<double> &output, MatrixMTL<double> &msOutput,
    VectorMTL<double> &prodsurOutput, VectorMTL<double> &conssurOutput);

  VectorMTL<double> optInv(VectorMTL<double> &Vold, VectorMTL<double> &d);

#if TRANSITION_MATRIX == 0
  void valueIt(VectorMTL<double> &profit, VectorMTL<double> &iota1, VectorMTL<double> &rho1, VectorMTL<double> &V1, 
    MatrixMTL<double> &tranmat1, VectorMTL<double> &prcont1, VectorMTL<double> &d, VectorMTL<double> &condexp);
#elif TRANSITION_MATRIX == 1
  void valueIt(VectorMTL<double> &profit, VectorMTL<double> &iota1, VectorMTL<double> &rho1, VectorMTL<double> &V1, 
    TriDiagMatrixMTL<double> &tranmat1, VectorMTL<double> &prcont1, VectorMTL<double> &d, VectorMTL<double> &condexp);
#elif TRANSITION_MATRIX == 2
  void valueIt(VectorMTL<double> &profit, VectorMTL<double> &iota1, VectorMTL<double> &rho1, VectorMTL<double> &V1, 
    SparseMatrixMTL<double> &tranmat1, VectorMTL<double> &prcont1, VectorMTL<double> &d, VectorMTL<double> &condexp);
#else 
#error Invalid value for TRANSITION_MATRIX
#endif

#if TRANSITION_MATRIX == 0 
  void expState(MatrixMTL<double> &contTranMatrixMTL, VectorMTL<double> &tildes, 
    VectorMTL<double> &expfreq, VectorMTL<double> &d);
#elif TRANSITION_MATRIX == 1 
  void expState(TriDiagMatrixMTL<double> contTranMatrixMTL, VectorMTL<double> &tildes, 
    VectorMTL<double> &expfreq, VectorMTL<double> &d);
#elif TRANSITION_MATRIX == 2 
  void expState(SparseMatrixMTL<double> contTranMatrixMTL, VectorMTL<double> &tildes, 
    VectorMTL<double> &expfreq, VectorMTL<double> &d);
#else 
    #error Invalid value for TRANSITION_MATRIX
#endif

  int fpstrat(VectorMTL<double> &iota1, VectorMTL<double> &rho1, VectorMTL<double> &V1, 
    VectorMTL<double> &prof, VectorMTL<double> &expfreq, VectorMTL<double> &tildes, 
    VectorMTL<double> &d, VectorMTL<double> &);

  void fpstrat_gauss_seidel(VectorMTL<double> &iota1, VectorMTL<double> &rho1, VectorMTL<double> &V1,
    VectorMTL<double> &prof, VectorMTL<double> &expfreq, VectorMTL<double> &tildes,
    VectorMTL<double> &d, VectorMTL<double> &);


  void fpstratxmax(VectorMTL<double> *&iota, VectorMTL<double> *&rho, VectorMTL<double> *&V, VectorMTL<double> *&prof, 
    VectorMTL<double> *&tildes, VectorMTL<double> *&d, VectorMTL<double> *&);

  int compute(int &xmax,VectorMTL<double> &,VectorMTL<double> &);
  void computeBounds(double *, VectorMTL<double> &, VectorMTL<double> &, 
   VectorMTL<double> &, VectorMTL<double> &, double, VectorMTL<double> &);
  ~CompOE();

  int sample_constentr(RanGenLib &rnd, double lambda_local, MatrixMTL<double> &Prob, 
    double p,VectorMTL<double> &state);

  void concentrationRatios(VectorMTL<double> &ms, VectorMTL<double> &tildes, 
    double &c1, double &c4, double &c10, double &c20);
  void computeStats(double tol, int n_min, int n_max, VectorMTL<double> &iota, VectorMTL<double> &rho, 
    VectorMTL<double> &tildes, VectorMTL<double> &pricesStart, double lambda,  VectorMTL<double> &pricesOut,
    VectorMTL<double> &msOut, double &prodsurOut, double &conssurOut, double &totalsurOut, VectorMTL<double> &cOut,
    double &entrateOut, double &exirateOut, double &msexitOut);

  inline int getTransitionSize() {
  #if TRANSITION_MATRIX == 0
    return xmax*xmax;
  #elif TRANSITION_MATRIX == 1
    return 4+(xmax-2)*3;
  #elif TRANSITION_MATRIX == 2
    return 0;
  #else 
  #error Invalid value for TRANSITION_MATRIX
  #endif
  }

  void tranProbACW(MatrixMTL<double> &RHS, MatrixMTL<double> &tranMatrix);
  void tranProbACW(MatrixMTL<double> &RHS, SparseMatrixMTL<double> &tranMatrix);
};

double binocdf(int x, int n, double b);
unsigned int binomial_coef(int a, int j);
double binopdf(int x, int n, double b);
double gammln(double xx);
double factln(int n);
double bico(int n, int k);
#endif
