/***************************************************************************
 *   Copyright (C) 2006 by Jeziorski, Weintraub, Benkard and Van Roy       *
 *   przemekj@stanford.edu                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <math.h>
#include <compoe/compoe.h>
#include <compoe/compoe_nonstat.h>

#ifdef WITH_AGGR
#include <compoe/compoe_aggr.h>
#endif

#include <sys/times.h>
#include <sys/types.h>
#include <sys/time.h>
#include <matrix3d.h>
#include <parameters.h>

using namespace std;
using namespace MTL;

int main(int argc, char *argv[]) {
  struct tms t,u,s,w,u_test;
  long r0,r1,r2,r3,r_test;
  CompOE *oe;
  VectorMTL<double> *iota = NULL;
  VectorMTL<double> *rho = NULL;
  VectorMTL<double> *V = NULL;
  VectorMTL<double> *tildes = NULL;
  VectorMTL<double> *prcont = NULL;
  VectorMTL<double> *prices_end = NULL;
  double lambda;

  double prof_fun[PROF_FUN_PARAM_NO];
  double prof_fun_ipopt[PROF_FUN_IPOPT_N];
  double transition[TRANSITION_N];
  double constants[CONSTANTS_N];
  double init[INIT_N]; 
  double nonstatparams[NONSTAT_N];
  double convtol[CONVTOL_N];
  double bounds[BOUNDS_N];
  double control[CONTROL_N];
  double aggr[AGGR_N];

  initializeStructures(control, prof_fun, prof_fun_ipopt, transition, 
    constants, init, convtol, bounds, nonstatparams, aggr);
  int xmax = int(init[0]);

  ofstream myfile;
  myfile.open ("output");

  oe = new CompOE();
  oe->setOutput(&myfile);
  oe->initialize(control, prof_fun, prof_fun_ipopt, transition, constants, init, convtol);

  myfile << "Oblivious Equilibrium Solver" << endl;
  myfile << "----------------------------" << endl;

  myfile << "\nProfit Function Parameters:" << endl;
  for(int ii=0;ii<PROF_FUN_PARAM_NO;ii++) myfile << "  " << ii << ": " << prof_fun[ii] << endl;

//  myfile << "\nProfit Function Parameters for Ipopt:" << endl;
//  for(int ii=0;ii<PROF_FUN_IPOPT_N;ii++) myfile << "  " << ii << ": " << prof_fun_ipopt[ii] << endl;
 
  myfile << "\n\nDynamic Parameters:\n\n";
  for(int ii=0;ii<CONSTANTS_N;ii++) myfile << "  " << ii << ": " << constants[ii] << endl;

  myfile << "\n\nTransition Parameters:\n\n";
  for(int ii=0;ii<TRANSITION_N;ii++) myfile << "  " << ii << ": " << transition[ii] << endl;

  if(control[2]>0) {
    myfile << "  Fixed Number of Firms (No entry/No exit): " << control[2] << endl;
  } else {
    if(control[1]==1) {
        myfile << "\n  Poisson Entry Process" << endl;
    } else {
        myfile << "\n  Constant Entry Process" << endl;
    }
  }

  myfile << "\nInitial Conditions:\n\n";
  for(int ii=0;ii<INIT_N;ii++) myfile << "  " << ii << ": " << init[ii] << endl;

  r1 = times(&t);
  myfile << "\nSolving for OE...\n";


  VectorMTL<double> iotaInit(xmax);
  VectorMTL<double> rhoInit(xmax);

  iotaInit=init[1];
  rhoInit=init[2];

//  if(oe->compute(xmax,rhoInit,iotaInit)==0) {
//    cout << "Cannot compute OE\n\n" << endl;
//    myfile << "Cannot compute OE\n\n" << endl;
//    exit(0);
//  }

  r2 = times(&u);

  oe->setMethod(1);
  if(oe->compute(xmax,rhoInit,iotaInit)==0) {
    cout << "Cannot compute OE\n\n" << endl;
    myfile << "Cannot compute OE\n\n" << endl;
    exit(0);
  }


  r_test = times(&u_test);
  V = oe->getV();
  iota = oe->getIota();
  rho = oe->getRho();
  lambda = oe->getLambda();
  tildes = oe->getTildes();
  xmax = oe->getXmax();
  prcont = oe->getPrcont();
  prices_end = oe->getPrices();


  myfile << "\nOblivious Equilibrium Statistics:" << endl;
  myfile << "  ---------------------------------" << endl << endl;
  myfile << "  Oblivious Entry rate: " << lambda << endl;
  myfile << "  Expected number of firms: " << sum(*tildes)<< endl;


  myfile << "\n\nCondition number: " << oe->getCondNumber() <<endl;
  #ifdef BOUNDS
    oe->computeBounds(bounds, *iota, *rho, *V, *tildes, lambda, *prices_end);
  #endif
  r3 = times(&s);
  // Calculate profit statistics at oe
   VectorMTL<double> ms(xmax),prices(xmax);
   VectorMTL<double> profstate(xmax);
   double prodsur,conssur;

   if(oe->compProfAndStats(*tildes,*prices_end,prices,profstate,ms,prodsur,conssur)==0) {
     cout << "Cannot compute profit for statistics\n" << endl;
   } else { 
//     cout << *tildes;
//     cout << prices;
     // Calculate concetration ratios
     double c1=0, c4=0, c10=0, c20=0;
     oe->concentrationRatios(ms, *tildes, c1, c4, c10, c20);

     myfile << "\nConcentration ratios at expected state:\n\n";
     myfile << "  C1: " << c1 << endl;
     myfile << "  C4: " << c4 << endl;
     myfile << "  C10: " << c10 << endl;
     myfile << "  C20: " << c20 << endl;
     if(control[2]==0) {
       myfile << " \nExpected market share of entrants at expected state: "<< ms[int(constants[3])]*lambda <<endl;
     }
   }
  #if TRANSITION_MATRIX == 0 
     MatrixMTL<double> tranMatrixMTL(xmax);
     MatrixMTL<double> conttranMatrixMTL(xmax);
  #elif TRANSITION_MATRIX == 1 
     TriDiagMatrixMTL<double> tranMatrixMTL(xmax);
     TriDiagMatrixMTL<double> conttranMatrixMTL(xmax);
  #elif TRANSITION_MATRIX == 2 
     SparseMatrixMTL<double> tranMatrixMTL;
     SparseMatrixMTL<double> conttranMatrixMTL;
  #else 
    #error Invalid value for TRANSITION_MATRIX
  #endif

   oe->tranProb(*iota,tranMatrixMTL,conttranMatrixMTL,*prcont);
   myfile << "\nState\t Stilde\t Investment\t Prob Succ\t Prob Exit\n";
   for(int i=0;i<xmax-1;i++) {
       myfile << i << "\t" << (*tildes)[i] << "\t\t" <<  
       #if TRANSITION_MATRIX == 0 
          (*iota)[i] << "\t" << tranMatrixMTL[i][i+1] << "\t" << 1-(*prcont)[i] << endl;
       #elif TRANSITION_MATRIX == 1 
          (*iota)[i] << "\t" << tranMatrixMTL[-1][i] << "\t" << 1-(*prcont)[i] << endl;
       #elif TRANSITION_MATRIX == 2
          (*iota)[i] << "\t" << "-" << "\t" << 1-(*prcont)[i] << endl;
       #else 
          #error Invalid value for TRANSITION_MATRIX
       #endif
   }
   myfile << xmax-1 << "\t" << (*tildes)[xmax-1] << "\t\t" <<  (*iota)[xmax-1] << "\t0\t" << 1-(*prcont)[xmax-1] << endl;

   #ifdef BOUNDS
     // Output of the bound calculation and industry statistics

     double bound1avg, bound1pre, bound1var; 
     VectorMTL<double> bound1por(xmax); 
     VectorMTL<double> bound2avg(xmax), bound2pre(xmax), bound2var(xmax), bound2por(xmax); 
     VectorMTL<double> bound3avg(xmax), bound3pre(xmax), bound3var(xmax), bound3por(xmax); 
     VectorMTL<double> bound4avg(xmax), bound4pre(xmax), bound4var(xmax), bound4por(xmax); 
     oe->getBound1(bound1avg, bound1pre, bound1var, bound1por);
     oe->getBound2(bound2avg, bound2pre, bound2var, bound2por);
     oe->getBound3(bound3avg, bound3pre, bound3var, bound3por);
     oe->getBound4(bound4avg, bound4pre, bound4var, bound4por);

     myfile << "\n Bound 1 (Loosest bound -- uniform across x)" << endl;
     myfile << "---------------------------------------------" << endl;
     myfile << "State\t Bound\t /EV(x,s)\t Rel.Prec." << endl;
     for(int i=0;i<xmax;i++) {
       myfile << i << "\t" << bound1avg << "\t" <<  bound1por[i] << "\t" << bound1pre << endl;
     }

     myfile << "\n Bound 2 (Tighter bound -- no restrictions on investment process)" << endl;
     myfile << "---------------------------------------------" << endl;
     myfile << "State\t Bound\t /EV(x,s) \t Rel.Prec." << endl;
     for(int i=0;i<xmax;i++) {
       myfile << i << "\t" << bound2avg[i] << "\t" <<  bound2por[i] << "\t" << bound2pre[i] << endl;
     }

     myfile << "\n Bound 3 (Tightest bound -- investment process in (-1,0,1)" << endl;
     myfile << "---------------------------------------------" << endl;
     myfile << "State\t Bound\t /EV(x,s)\t Rel.Prec." << endl;
     for(int i=0;i<xmax;i++) {
       myfile << i << "\t" << bound3avg[i] << "\t" <<  bound3por[i] << "\t" << bound3pre[i] << endl;
     }

     myfile << "\n Bound 4 (Even Tighter -- investment process in (-1,0,1)" << endl;
     myfile << "---------------------------------------------" << endl;
     myfile << "State\t Bound\t /EV(x,s)\t Rel.Prec." << endl;
     for(int i=0;i<xmax;i++) {
       myfile << i << "\t" << bound4avg[i] << "\t" <<  bound4por[i] << "\t" << bound4pre[i] << endl;
     }


     myfile << "\n Industry Statistics \n" << endl;
     myfile << "Expected value wrt invariant distr. of industry state (standard dev.)" << endl;
     myfile << "---------------------------------------------------------------------" << endl;
     myfile << "Entry rate (number of entrants/total number of firms): " << 
       oe->getEntrateavg() << " (" << oe->getEntratevar() << ")" <<endl;
     myfile << "Exit rate (number of exiters/total number of firms): " <<
       oe->getExirateavg() << " (" << oe->getExiratevar() << ")" <<endl;
     myfile << "Market share exiters: " <<
       oe->getMsexitavg() << " (" << oe->getMsexitvar() << ")" <<endl;
     myfile << "Producer surplus: " <<
       oe->getProdsuravg() << " (" << oe->getProdsurvar() << ")" <<endl;
     myfile << "Consumer surplus: " <<
       oe->getConssuravg() << " (" << oe->getConssurvar() << ")" <<endl;
     myfile << "Total surplus: " <<
       oe->getTotalsuravg() << " (" << oe->getTotalsurvar() << ")" <<endl;
     myfile << "C1: " <<
       oe->getC1avg() << " (" << oe->getC1var() << ")" <<endl;
     myfile << "C4: " <<
       oe->getC4avg() << " (" << oe->getC4var() << ")" <<endl;
     myfile << "C10: " <<
       oe->getC10avg() << " (" << oe->getC10var() << ")" <<endl;
     myfile << "C20: " <<
       oe->getC20avg() << " (" << oe->getC20var() << ")" <<endl;
   #endif
//   myfile << "\nExecution times: \n\tEquilibrium calculation \t" << r2-r1 << endl;
//   myfile << "\tBounds calculation \t\t" << r3-r2 << endl;
//   myfile << "\tTotal time \t\t\t" << r3-r1 << endl;
   cout << "\nExecution times: \n\tEquilibrium calculation \t" << r2-r1 << endl;
//  cout << "\tBounds calculation \t\t" << r3-r2 << endl;
//   cout << "\tTotal time \t\t\t" << r3-r1 << endl;

   cout << "\tTest \t\t\t\t" << r_test-r2 << endl;

   // Non stationaty part
   #ifdef NONSTAT
     CompOE_nonstat *oe_nonstat = new CompOE_nonstat;
     oe_nonstat->setOutput(&myfile);
     oe_nonstat->initialize(control, prof_fun, prof_fun_ipopt, transition, constants, init, convtol, nonstatparams);


     int Tbar,xmax_nonstat;
//     oe_nonstat->computeT(V,iota,rho,lambda,Tbar,xmax_nonstat);

     Tbar=int(nonstatparams[10]);

     if(int(nonstatparams[12]) != 0) {
       xmax_nonstat = int(nonstatparams[12]);
       cout << endl << endl << "Recompute the oblivious equilibrium for xmax = " << xmax_nonstat << endl;
       init[0] = xmax_nonstat;
       VectorMTL<double> iotaInit_temp(xmax_nonstat);
       VectorMTL<double> rhoInit_temp(xmax_nonstat);

       iotaInit_temp=init[1];
       rhoInit_temp=init[2];

       oe->initialize(control, prof_fun, prof_fun_ipopt, transition, constants, init, convtol);
       oe->compute(xmax_nonstat,rhoInit_temp,iotaInit_temp);
     } else {
       xmax_nonstat = xmax;
     }

     // Construct initial vectors 
     VectorMTL<double> *V_nonstat = oe->getV();
     VectorMTL<double> *iota_nonstat = oe->getIota();
     VectorMTL<double> *rho_nonstat = oe->getRho();
     VectorMTL<double> *prices_nonstat = oe->getPrices();
     VectorMTL<double> *tildes_nonstat = oe->getTildes();
     double lambda_nonstat = oe->getLambda();


     /*** Construct s0 from setup.h ***/
     double s0_init_data[] = S0_NONSTAT;
     VectorMTL<double> s0_init(xmax_nonstat);
     for(int i=0;i<S0_SIZE;i++) {
       s0_init[i]=s0_init_data[i];
     }
     for(int i=S0_SIZE;i<xmax_nonstat;i++) {
       s0_init[i]=nonstatparams[8];
     }

// Uncomment if you want to start from OE
     s0_init=*tildes;

     cout << "\n\nNonstationary part:\n\t Tbar = " << Tbar << "\n\t xmax = " << xmax_nonstat << endl;
     myfile << "\n\nNonstationary part:\n\t Tbar = " << Tbar << "\n\t xmax = " << xmax_nonstat << endl;
     r0 = times(&w);
     oe_nonstat->main_loop(xmax_nonstat, Tbar, *V_nonstat, *iota_nonstat,
       *rho_nonstat, s0_init, *tildes_nonstat, *prices_nonstat, lambda_nonstat);

     MatrixMTL<double> *sMatrix = oe_nonstat->getsMatrix();

//     cout << "State matrix\n" << endl;
//     cout << *sMatrix << endl << endl;
     myfile << "State matrix\n" << endl;
     myfile << *sMatrix << endl << endl;

     MatrixMTL<double> *VMatrix = oe_nonstat->getVMatrix();

//     cout << "Value function matrix\n" << endl;
//     cout << *VMatrix << endl << endl;
     myfile << "Value function matrix\n" << endl;
     myfile << *VMatrix << endl << endl;

     MatrixMTL<double> *iotaMatrix = oe_nonstat->getIotaMatrix();

//     cout << "Iota matrix\n" << endl;
//     cout << *iotaMatrix << endl << endl;
     myfile << "Iota matrix\n" << endl;
     myfile << *iotaMatrix << endl << endl;;

     MatrixMTL<double> *rhoMatrix = oe_nonstat->getRhoMatrix();

//     cout << "Rho matrix\n" << endl;
//     cout << *rhoMatrix << endl << endl;
     myfile << "Rho matrix\n" << endl;
     myfile << *rhoMatrix << endl << endl;

     MatrixMTL<double> *pMatrix = oe_nonstat->getpMatrix();

//     cout << "Prices matrix\n" << endl;
//     cout << *pMatrix << endl << endl;
     myfile << "Prices matrix\n" << endl;
     myfile << *pMatrix << endl << endl;

     VectorMTL<double> *lambdaVector = oe_nonstat->getLambdaVector();

//     cout << "Lambda Vector\n" << endl;
//     cout << *lambdaVector << endl << endl;
     myfile << "Lambda Vector\n" << endl;
     myfile << *lambdaVector << endl << endl;

     r1 = times(&t);

     #ifdef BOUNDS_NONSTAT
       oe_nonstat->computeBoundsNonstat(bounds, *iotaMatrix, *rhoMatrix, *sMatrix, *lambdaVector, *pMatrix);

       VectorMTL<double> bound1avg_nonstat(xmax), bound1pre_nonstat(xmax), 
         bound1var_nonstat(xmax); 
       oe_nonstat->getBound1_nonstat(bound1avg_nonstat, bound1pre_nonstat, bound1var_nonstat);
       myfile << "\n\n Bound for nonstationary OE 1 (no restrictions on investment process)" << endl;
       myfile << "---------------------------------------------" << endl;
       myfile << "State\t Bound \t Rel.Prec." << endl;
       for(int i=0;i<xmax;i++) {
         myfile << i << "\t" << bound1avg_nonstat[i] << "\t" << "\t" 
           << bound1pre_nonstat[i] << endl;
       }

       VectorMTL<double> bound2avg_nonstat(xmax), bound2pre_nonstat(xmax), 
         bound2var_nonstat(xmax); 
       oe_nonstat->getBound2_nonstat(bound2avg_nonstat, bound2pre_nonstat, bound2var_nonstat);
       myfile << "\n\n Bound for nonstationary OE 1 (no restrictions on investment process)" << endl;
       myfile << "---------------------------------------------" << endl;
       myfile << "State\t Bound \t Rel.Prec." << endl;
       for(int i=0;i<xmax;i++) {
         myfile << i << "\t" << bound2avg_nonstat[i] << "\t" << "\t" 
           << bound2pre_nonstat[i] << endl;
       }
     #endif
     r2 = times(&s);

     delete oe_nonstat;
     myfile << "\nExecution times: \n\tNostationary Equilibrium calculation \t" << r1-r0 << endl;
     myfile << "\tNostationary Bounds calculation \t" << r2-r1 << endl;
     myfile << "\tTotal time \t\t\t\t" << r2-r0 << endl;
     cout << "\nExecution times: \n\tNostationary Equilibrium calculation \t" << r1-r0 << endl;
     cout << "\tNostationary Bounds calculation \t" << r2-r1 << endl;
     cout << "\tTotal time \t\t\t\t" << r2-r0 << endl;
  #endif

  #ifdef AGGR_OE
    #ifdef WITH_AGGR
      int xmax_nonstat = xmax;

      CompOE_aggr *oe_aggr = new CompOE_aggr;
      ofstream myfile_aggr;

      myfile_aggr.open ("output_aggr");
      oe_aggr->setOutput(&myfile_aggr);

      SparseMatrixMTL<double> shockTran;
      shockTran.bandMatrix(3,3);

      // From Low
      shockTran[0] = 0.5; // (0,0)
      shockTran[1] = 0.5; // (0,1)

      // From Medium
      shockTran[2] = 0.25; // (1,0)
      shockTran[3] = 0.5;  // (1,1)
      shockTran[4] = 0.25;  // (1,2)

      // From High
      shockTran[5] = 0.5; // (2,1)
      shockTran[6] = 0.5; // (2,2)

      int shock_states = shockTran.getNRows();

      oe_aggr->initialize(control, prof_fun, prof_fun_ipopt, transition, constants, init, convtol, aggr);
      oe_aggr->compute_aggr(xmax_nonstat,*iota,*rho,lambda,shockTran);

      myfile_aggr << "\nAggregate Shocks Oblivious Equilibrium Statistics:" << endl;
      myfile_aggr << "  ---------------------------------" << endl << endl;
      myfile_aggr << "  Oblivious Entry rates: \n";

      VectorMTL<double> *lambdaVector;
      lambdaVector=oe_aggr->getLambdaAggr();

//       for(int w=0;w<shock_states;w++) {
//         myfile_aggr << "    Shock: " << w << ", Lambda: " << (*lambdaVector)[w] << endl;
//       }

/*      myfile_aggr << "\n  Expected number of firms: " << endl;*/
      MatrixMTL<double> *tildesMatrix;
      tildesMatrix=oe_aggr->getTildesAggr();

//       for(int w=0;w<shock_states;w++) {
//         myfile_aggr << "    Shock: " << w << ", Value: " << sum(tildesMatrix->getRow(w)) << endl;
//       }

/*      myfile_aggr << "\n  Value function: " << endl;*/
      MatrixMTL<double> *VMatrix;
      VMatrix=oe_aggr->getVAggr();
//       myfile_aggr << (*VMatrix);

/*      myfile_aggr << "\n  Expected value function: " << endl;*/
      MatrixMTL<double> *VExpectedMatrix;
      VExpectedMatrix=oe_aggr->getVExpectedAggr();
//       myfile_aggr << (*VExpectedMatrix);


/*      myfile_aggr << "\n  Expected state: " << endl;*/
//       myfile_aggr << (*tildesMatrix);

      #ifdef BOUNDS_AGGR_OE

      MatrixMTL<double> *iotaMatrix_aggr=oe_aggr->getIotaAggr();
      MatrixMTL<double> *rhoMatrix_aggr=oe_aggr->getRhoAggr();
      MatrixMTL<double> *pricesMatrix_aggr=oe_aggr->getPricesInitAggr();

      oe_aggr->computeBounds_aggr(bounds, *iotaMatrix_aggr, *rhoMatrix_aggr, 
        *tildesMatrix, *lambdaVector, *pricesMatrix_aggr, shockTran);

       VectorMTL<double> bound1avg_aggr(xmax), bound1pre_aggr(xmax), 
         bound1var_aggr(xmax); 
       oe_aggr->getBound1_aggr(bound1avg_aggr, bound1pre_aggr, bound1var_aggr);
       myfile_aggr << "\n\n Bound1 for OE with aggregate shocks " << endl;
       myfile_aggr << "---------------------------------------------" << endl;
       myfile_aggr << "State\t Bound \t Rel.Prec." << endl;
       for(int i=0;i<xmax;i++) {
         myfile_aggr << i << "\t" << bound1avg_aggr[i] << "\t" << "\t" 
           << bound1pre_aggr[i] << endl;
       }

       VectorMTL<double> bound2avg_aggr(xmax), bound2pre_aggr(xmax), 
         bound2var_aggr(xmax); 
       oe_aggr->getBound2_aggr(bound2avg_aggr, bound2pre_aggr, bound2var_aggr);
       myfile_aggr << "\n\n Bound2 for OE with aggregate shocks" << endl;
       myfile_aggr << "---------------------------------------------" << endl;
       myfile_aggr << "State\t Bound \t Rel.Prec." << endl;
       for(int i=0;i<xmax;i++) {
         myfile_aggr << i << "\t" << bound2avg_aggr[i] << "\t" << "\t" 
           << bound2pre_aggr[i] << endl;
       }

      #endif

      delete oe_aggr;
      myfile_aggr.close();
    #endif
  #endif
  delete oe;

  myfile.close();
  return EXIT_SUCCESS;
}
